//
//  bdlive_viewer_player_impl.dart
//
//  bdlive-flutter-viewer SDK License
//
//  Copyright 2025 Byteplus Pte. Ltd. All Rights Reserved.
//
//  The bdlive-flutter-viewer SDK was developed by Byteplus Pte. Ltd. (hereinafter “Byteplus”).
//  Any copyright or patent right is owned by and proprietary material of the Byteplus.
//
//  bdlive-flutter-viewer SDK is available under the Byteplus product and licensed under the commercial license.
//  Customers can contact via https://www.byteplus.com/en/contact for commercial licensing options.
//  Here is also a link to Service Level Agreement: https://docs.byteplus.com/en/byteplus-livesaas/docs/byteplus-live-service-level-agreement
//
//  Without Byteplus's prior written permission, any use of bdlive-flutter-viewer SDK, in particular any use for commercial purposes, is prohibited.
//  This includes, without limitation, incorporation in a commercial product, use in a commercial service, or production of other artefacts for commercial purposes.
//
//  Without Byteplus's prior written permission, the bdlive-flutter-viewer SDK may not be reproduced, modified and/or made available in any form to any third party.
import 'package:bdlive_flutter_viewer/src/bdlive_viewer_manager_impl.dart';
import 'package:flutter/foundation.dart';
import 'package:flutter/services.dart';

import '../api/bdlive_viewer_player_api.dart';
import '../api/bdlive_viewer_player_delegate.dart';
import '../api/bdlive_model.dart';
import 'base/bdlive_event_channel.dart';
import 'bdlive_viewer_const.dart';
import 'bdlive_viewer_player_delegate_impl.dart';
import 'base/bdlive_channel_extension.dart';
import 'base/bdlive_enum_convert.dart';


class BDLiveViewerPlayerImpl implements BdliveViewerPlayer {

  static int _incrementId = 0;

  @override
  final int managerId;

  final MethodChannel _channel;

  RTCEventChannel? _observerChannel;

  BDLiveViewerPlayerImpl({required this.managerId}): _channel = MethodChannel('$LIVE_PLAYER_CHANNEL$managerId');

  static Future<BdliveViewerPlayer> createPlayer(BdlViewerPlayerConfig activity) async {
    _incrementId += 1;
    BDLiveViewerPlayerImpl player = BDLiveViewerPlayerImpl(managerId: _incrementId);
    Map<String, dynamic> map = activity.toMap();
    map['managerId'] = _incrementId;
    await BdliveViewerManagerImpl.staticChannel.bdliveInvokeMethod('createPlayer', map);
    return player;
  }

  // 启动直播间纯净播放器（开始进房+鉴权）
  @override
  Future<void> start() {
    return _channel.bdliveInvokeMethod('start');
  }

  // 播放
  @override
  Future<void> play() {
    return _channel.bdliveInvokeMethod('play');
  }

  // 暂停播放
  @override
  Future<void> pause() {
    return _channel.bdliveInvokeMethod('pause');
  }

  //设置回调
  @override
  Future<void> setDelegate(BdliveViewerPlayerDelegate delegate) async {
    _observerChannel?.cancel();
    var result = await _channel.bdliveInvokeMethod("setDelegate");
    RTCEventChannel channel =
    RTCEventChannel('$LIVE_PLAYER_EVENT_CHANNEL$managerId');
    channel.listen((String methodName, Map<dynamic, dynamic> dic) {
      debugPrint('dart - listen = $dic');
      delegate.process(methodName, dic);
    });
    _observerChannel = channel;
    return result;
  }

  //刷新当前直播内容。在 `playableStatus` 的值为 `2` 时可调用此方法。
  @override
  Future<void> refreshLive() {
    return _channel.bdliveInvokeMethod('refreshLive');
  }

  // 获取是否正在播放视频
  Future<bool?> isPlaying() {
    return _channel.bdliveInvokeMethod<bool>('isPlaying');
  }

  // 获取当前视频是否卡顿
  @override
  Future<bool?> isStalling() {
    return _channel.bdliveInvokeMethod<bool>('isStalling');
  }

  // 获取当前资源的可播放状态
  @override
  Future<BdlivePlayableStatus> getPlayableStatus() {
    var result = _channel.bdliveInvokeMethod<int>('getPlayableStatus');
    return result.then((value) => value.playableStatus);
  }

  // 设置视频画面填充模式
  @override
  Future<void> setPlayerLayoutMode(BdlivePlayerLayoutMode playerLayoutMode) {
    return _channel.bdliveInvokeMethod('setPlayerLayoutMode', {'playerLayoutMode' : playerLayoutMode.index});
  }

  @override
  Future<BdlivePlayerLayoutMode?> getPlayerLayoutMode() {
    var result = _channel.bdliveInvokeMethod<int>('getPlayerLayoutMode');
    return result.then((value) => value.playerLayoutMode);
  }

  // 获取当前支持的分辨率列表
  @override
  Future<List<BdliveVideoResolution>?> getResolutions() {
     final result = _channel.bdliveInvokeMethod('getResolutions');
     return result.then((value) {
       List<int> resolutions = value.cast<int>();
       return resolutions.map((ele) => ele.videoResolution).toList();
     });
  }

  // 获取当前选中的分辨率
  @override
  Future<BdliveVideoResolution> getCurResolution() {
    var result = _channel.bdliveInvokeMethod<int>('getCurResolution');
    return result.then((value) => value.videoResolution);
  }

  // 为当前视频设置分辨率
  @override
  Future<void> setCurResolution(BdliveVideoResolution resolution) {
    return _channel.bdliveInvokeMethod('setCurResolution', {'resolution' : resolution.index});
  }

  // 获取当前点播视频的播放进度（单位：ms）
  @override
  Future<int?> getCurVodPlayTime() {
    return _channel.bdliveInvokeMethod<int>('getCurVodPlayTime');
  }

  // 获取当前点播视频的时长（单位：ms）
  @override
  Future<int?> getCurVodDuration() {
    return _channel.bdliveInvokeMethod<int>('getCurVodDuration');
  }

  // 跳转到点播视频的指定播放进度（单位：ms）
  @override
  Future<bool?> seekVodTime(int timeInMills) {
    return _channel.bdliveInvokeMethod<bool>('seekVodTime', {'timeInMills' : timeInMills});
  }

  // 设置是否静音播放。默认不静音
  @override
  Future<void> setMute(bool isMute) {
    return _channel.bdliveInvokeMethod('setMute', {'isMute' : isMute});
  }

  // 设置点播视频是否循环播放
  @override
  Future<void> setVodLoop(bool isLoop) {
    return _channel.bdliveInvokeMethod('setVodLoop', {'isLoop' : isLoop});
  }

  // playSpeed 播放倍速。取值范围：(0,2]
  // 设置点播视频播放倍速
  @override
  Future<void> setVodPlaySpeed(double playSpeed) {
    return _channel.bdliveInvokeMethod('setVodPlaySpeed', {'playSpeed' : playSpeed});
  }

  // 当前点播视频的播放倍速。取值范围：(0,2]。如果当前未播放点播视频，则返回 `-1` 。
  //获取当前点播视频的播放倍速
  @override
  Future<double?> getCurVodPlaySpeed() {
    return _channel.bdliveInvokeMethod<double>('getCurVodPlaySpeed');
  }

  // 当独立播放器所在的页面进入后台运行时，可调用此方法通知播放器。
  @override
  Future<void> setInBackground(bool inBackground) {
    return _channel.bdliveInvokeMethod('setInBackground', {'inBackground': inBackground});
  }

  // 获取当前直播间的状态。
  @override
  Future<BdLiveRoomStatus> getLiveRoomStatus() {
    var result = _channel.bdliveInvokeMethod<int>('getLiveRoomStatus');
    return result.then((value) => value.liveRoomStatus);
  }

  // 设置是否强制隐藏封面图
  @override
  Future<void> setCoverImageContainerVisibility(bool visible) {
    return _channel.bdliveInvokeMethod('setCoverImageContainerVisibility', {'visible' : visible});
  }

  @override
  Future<void> destroy() {
   return _channel.bdliveInvokeMethod('destroy');
  }
}

