//
//  bdlive_flutter_viewer_impl.dart
//
//  bdlive-flutter-viewer SDK License
//
//  Copyright 2025 Byteplus Pte. Ltd. All Rights Reserved.
//
//  The bdlive-flutter-viewer SDK was developed by Byteplus Pte. Ltd. (hereinafter “Byteplus”).
//  Any copyright or patent right is owned by and proprietary material of the Byteplus.
//
//  bdlive-flutter-viewer SDK is available under the Byteplus product and licensed under the commercial license.
//  Customers can contact via https://www.byteplus.com/en/contact for commercial licensing options.
//  Here is also a link to Service Level Agreement: https://docs.byteplus.com/en/byteplus-livesaas/docs/byteplus-live-service-level-agreement
//
//  Without Byteplus's prior written permission, any use of bdlive-flutter-viewer SDK, in particular any use for commercial purposes, is prohibited.
//  This includes, without limitation, incorporation in a commercial product, use in a commercial service, or production of other artefacts for commercial purposes.
//
//  Without Byteplus's prior written permission, the bdlive-flutter-viewer SDK may not be reproduced, modified and/or made available in any form to any third party.
import 'dart:collection';

import 'package:bdlive_flutter_viewer/src/bdlive_viewer_manager_impl.dart';
import 'package:bdlive_flutter_viewer/api/config/bdlive_viewer_config.dart';
import 'package:flutter/foundation.dart';
import 'package:flutter/material.dart';
import '../api/bdlive_flutter_viewer_api.dart';
import '../api/bdlive_model.dart';
import 'base/bdlive_event_channel.dart';
import 'bdlive_room_status_listener_extension.dart';

class BdliveFlutterViewerImpl implements BdliveViewerApi {
  RTCEventChannel? _observerChannel;
  BdlLiveRoomStatusListener? _listener;
  BdLiveViewerConfig _config = BdLiveViewerConfig();

  @override
  BdLiveViewerConfig getLiveRoomConfig() {
    return _config;
  }

  @override
  Future<void> setLiveRoomConfig(BdLiveViewerConfig config) async {
    _config = config;
    await _invokeMethod('setLiveRoomConfig', config.toMap());
  }

  @override
  Future<bool> enterLiveRoomLogically(BdlActivityConfig activity) async {
    var result =
        await _invokeMethod<bool>('enterLiveRoomLogically', activity.toMap());
    return result ?? false;
  }

  @override
  Future<void> leaveLiveRoomLogically() {
    return _invokeMethod('leaveLiveRoomLogically');
  }

  @override
  Future<bool> enterLiveRoom(BdlActivityConfig activity) async {
    var result = await _invokeMethod<bool>('enterLiveRoom', activity.toMap());
    return result ?? false;
  }

  @override
  Future<void> leaveLiveRoom() {
    return _invokeMethod('leaveLiveRoom');
  }

  @override
  Future<void> setLiveRoomStatusListener(
      BdlLiveRoomStatusListener? listener) async {
    if (listener == _listener) {
      return;
    }
    _listener = listener;
    _observerChannel?.cancel();
    var result = await _invokeMethod("setLiveRoomStatusListener", {
      'hasListener': listener != null,
    });
    if (listener == null) {
      return;
    }
    RTCEventChannel channel =
        RTCEventChannel('com.bytedance.flutter_viewer.listener.event');
    channel.listen((String methodName, Map<dynamic, dynamic> dic) {
      debugPrint('dart - listen = $dic  $listener');

      listener.process(methodName, dic);
    });
    _observerChannel = channel;
    return result;
  }

  @override
  Future<void> hideLiveRoom() {
    return _invokeMethod('hideLiveRoom');
  }

  @override
  Future<void> openUrl(
      String url, BdlViewerOpenUrlType type, bool enableFloating) {
    HashMap<String, dynamic> dic = HashMap();
    dic['url'] = url;
    dic['type'] = type.index + 1;
    dic['enableFloating'] = enableFloating;
    return _invokeMethod('openUrl', dic);
  }

  @override
  Future<void> closeFloatingView() {
    return _invokeMethod('closeFloatingView');
  }

  @override
  Future<void> dispatchWidgetSate(int state) {
    HashMap<String, dynamic> dic = HashMap();
    dic['state'] = state;
    return _invokeMethod("dispatchWidgetSate", dic);
  }

  @override
  Future<bool> onWillClose() {
    return _invokeMethod("onWillClose") as Future<bool>;
  }

  Future<T?> _invokeMethod<T>(String method, [dynamic arguments]) {
    return BdliveViewerManagerImpl.staticChannel
        .invokeMethod(method, arguments);
  }
}
