//
//  bdlive_viewer_player_api.dart
//
//  bdlive-flutter-viewer SDK License
//
//  Copyright 2025 Byteplus Pte. Ltd. All Rights Reserved.
//
//  The bdlive-flutter-viewer SDK was developed by Byteplus Pte. Ltd. (hereinafter “Byteplus”).
//  Any copyright or patent right is owned by and proprietary material of the Byteplus.
//
//  bdlive-flutter-viewer SDK is available under the Byteplus product and licensed under the commercial license.
//  Customers can contact via https://www.byteplus.com/en/contact for commercial licensing options.
//  Here is also a link to Service Level Agreement: https://docs.byteplus.com/en/byteplus-livesaas/docs/byteplus-live-service-level-agreement
//
//  Without Byteplus's prior written permission, any use of bdlive-flutter-viewer SDK, in particular any use for commercial purposes, is prohibited.
//  This includes, without limitation, incorporation in a commercial product, use in a commercial service, or production of other artefacts for commercial purposes.
//
//  Without Byteplus's prior written permission, the bdlive-flutter-viewer SDK may not be reproduced, modified and/or made available in any form to any third party.

import 'dart:collection';

import 'package:bdlive_flutter_viewer/src/bdlive_viewer_player_impl.dart';
import 'package:flutter/foundation.dart';

import 'bdlive_viewer_player_delegate.dart';
import 'bdlive_model.dart';

class BdlViewerPlayerConfig {
  // Live room id
  final int activityId;

  // Live room entry token
  final String token;

  // Live room authentication mode, see BdlViewerAuthMode for details
  final BdlViewerAuthMode authMode;

  BdlViewerPlayerConfig({
    required this.activityId,
    required this.token,
    required this.authMode,
  });

  Map<String, dynamic> toMap() {
    HashMap<String, dynamic> dic = HashMap();
    dic['activityId'] = activityId;
    dic['token'] = token;
    dic['authMode'] = authMode.index + 1;
    dic['isPortrait'] = false;
    return dic;
  }

  factory BdlViewerPlayerConfig.fromMap(Map<dynamic, dynamic> map) {
    debugPrint('BdlViewerPlayerConfig fromMap = $map');
    int activityId = map['activityId'] ?? 0;
    String token = map['token'] ?? '';
    int authMode = map['authMode'] ?? 0;
    return BdlViewerPlayerConfig(
        activityId: activityId,
        token: token,
        authMode: BdlViewerAuthMode.values[authMode - 1],
        );
  }

  @override
  toString() {
    return 'BdlViewerPlayerConfig {activityId: $activityId, token: $token, authMode: $authMode}';
  }
}


abstract class BdliveViewerPlayer {

  // 创建播放器
  static Future<BdliveViewerPlayer> createPlayer(BdlViewerPlayerConfig activity) => BDLiveViewerPlayerImpl.createPlayer(activity);

  // 启动直播间纯净播放器（开始进房+鉴权）
  Future<void> start();

  int get managerId;

  // 播放
  Future<void> play();

  // 暂停播放
  Future<void> pause();

  // 设置播放器状态回调
  Future<void> setDelegate(BdliveViewerPlayerDelegate delegate);

  // 刷新当前直播内容。在 `playableStatus` 的值为 `liveCanPlay` 时可调用此方法。
  Future<void> refreshLive();

  // 获取是否正在播放视频
  Future<bool?> isPlaying();

  // 获取当前视频是否卡顿
  Future<bool?> isStalling();

  // 获取当前资源的可播放状态
  Future<BdlivePlayableStatus> getPlayableStatus();

  // 设置视频画面填充模式
  Future<void> setPlayerLayoutMode(BdlivePlayerLayoutMode playerLayoutMode);

  // 获取视频画面填充模式
  Future<BdlivePlayerLayoutMode?> getPlayerLayoutMode();

  // 获取当前支持的分辨率列表
  Future<List<BdliveVideoResolution>?> getResolutions();

  // 获取当前选中的分辨率
  Future<BdliveVideoResolution> getCurResolution();

  // 为当前视频设置分辨率
  Future<void> setCurResolution(BdliveVideoResolution resolution);

  // 获取当前点播视频的播放进度（单位：ms）
  Future<int?> getCurVodPlayTime();

  // 获取当前点播视频的时长（单位：ms）
  Future<int?> getCurVodDuration();

  // 跳转到点播视频的指定播放进度（单位：ms）
  Future<bool?> seekVodTime(int timeInMills);

  // 设置是否静音播放。默认不静音
  Future<void> setMute(bool isMute);

  // 设置点播视频是否循环播放
  Future<void> setVodLoop(bool isLoop);

  // playSpeed 播放倍速。取值范围：(0,2]
  // 设置点播视频播放倍速
  Future<void> setVodPlaySpeed(double playSpeed);

  // 当前点播视频的播放倍速。取值范围：(0,2]。如果当前未播放点播视频，则返回 `-1` 。
  //获取当前点播视频的播放倍速
  Future<double?> getCurVodPlaySpeed();

  // 当独立播放器所在的页面进入后台运行时，可调用此方法通知播放器。
  Future<void> setInBackground(bool inBackground);

  // 获取当前直播间的状态。
  Future<BdLiveRoomStatus> getLiveRoomStatus();

  // 设置是否强制隐藏封面图
  Future<void> setCoverImageContainerVisibility(bool visible);

  // 销毁播放器
  Future<void> destroy();
}

