//
//  bdlive_model.dart
//
//  bdlive-flutter-viewer SDK License
//
//  Copyright 2025 Byteplus Pte. Ltd. All Rights Reserved.
//
//  The bdlive-flutter-viewer SDK was developed by Byteplus Pte. Ltd. (hereinafter “Byteplus”).
//  Any copyright or patent right is owned by and proprietary material of the Byteplus.
//
//  bdlive-flutter-viewer SDK is available under the Byteplus product and licensed under the commercial license.
//  Customers can contact via https://www.byteplus.com/en/contact for commercial licensing options.
//  Here is also a link to Service Level Agreement: https://docs.byteplus.com/en/byteplus-livesaas/docs/byteplus-live-service-level-agreement
//
//  Without Byteplus's prior written permission, any use of bdlive-flutter-viewer SDK, in particular any use for commercial purposes, is prohibited.
//  This includes, without limitation, incorporation in a commercial product, use in a commercial service, or production of other artefacts for commercial purposes.
//
//  Without Byteplus's prior written permission, the bdlive-flutter-viewer SDK may not be reproduced, modified and/or made available in any form to any third party.

import 'dart:collection';

import 'package:flutter/foundation.dart';

enum BdlViewerAuthMode {
  /**
   * In public mode, viewers enter the live broadcast room as visitors. In scenarios where user information is required,
   * such as clicking the comment input box or liking a comment,
   * they need to complete visitor registration (i.e. enter a nickname) first.
   */
  public,
  /**
   * In custom mode, viewers enter the live broadcast room using their user information in your app,
   * so they can directly send comments, like comments, etc.
   */
  custom,
}

enum BdlViewerOpenUrlType {
  // Page header ads
  pageHeaderAd,
  // Mid-page Ads
  pageAd,
  // Floating Ads
  floatingAd,
  // Product cards, including products in the product list and floating products
  cardAd,
}

class BdlActivityConfig {
  // Live room id
  final int activityId;

  // Live room entry token
  final String token;

  // Live room authentication mode, see BdlViewerAuthMode for details
  final BdlViewerAuthMode authMode;

  // Do you want to enter the horizontal or vertical version of the live broadcast room? true: vertical version; false: horizontal version
  final bool isPortrait;

  BdlActivityConfig({
    required this.activityId,
    required this.token,
    required this.authMode,
    required this.isPortrait,
  });

  Map<String, dynamic> toMap() {
    HashMap<String, dynamic> dic = HashMap();
    dic['activityId'] = activityId;
    dic['token'] = token;
    dic['authMode'] = authMode.index + 1;
    dic['isPortrait'] = isPortrait;
    return dic;
  }

  factory BdlActivityConfig.fromMap(Map<dynamic, dynamic> map) {
    debugPrint('BdlActivityConfig fromMap = $map');
    int activityId = map['activityId'] ?? 0;
    String token = map['token'] ?? '';
    int authMode = map['authMode'] ?? 0;
    bool isPortrait = map['isPortrait'] ?? false;
    return BdlActivityConfig(
        activityId: activityId,
        token: token,
        authMode: BdlViewerAuthMode.values[authMode - 1],
        isPortrait: isPortrait);
  }

  @override
  toString() {
    return 'BdlActivityConfig {activityId: $activityId, token: $token, authMode: $authMode, isPortrait: $isPortrait}';
  }
}

/// 视频画面填充模式
enum BdlivePlayerLayoutMode {
  /// 保持视频原有宽高比例填充播放器，视频的宽度或高度会填满播放器的宽度或高度。如果视频宽高比与播放器宽高比不同，视频会缩放显示。
  aspectFit,
  /// 视频内容完全填充播放器，但视频宽高比可能发生变化。
  fill,
  /// 保持视频原有宽高比例填充播放器，视频的宽高会填满播放器的宽高。如果视频宽高比与播放器宽高比不同，会有部分视频内容被裁剪掉。
  aspectFill,
}

/// 直播状态
enum BdLiveRoomStatus {
  /// 未知
  unknown,
  /// 直播中
  live,
  /// 预告
  preview,
  /// 回放
  replay,
  /// 已结束
  end,
}

enum BdlivePlayableStatus {
  /// 当前无任何资源可播放，无法调用 play 方法
  unknown,
  /// 当前有点播视频可播放，可以调用 play 方法播放视频。
  vodCanPlay,
  /// 当前有直播可播放，可以调用 play 方法播放视频。
  liveCanPlay,
}

class BdliveError {
  int? errorCode;
  String? errorMsg;

  BdliveError({required this.errorCode, required this.errorMsg});

  factory BdliveError.fromMap(Map<dynamic, dynamic> map) {
    int? errorCode = map['errorCode'];
    String? errorMsg = map['errorMsg'];
    return BdliveError(errorCode: errorCode, errorMsg: errorMsg);
  }

  @override
  String toString() {
    return "errorCode : $errorCode, errorMsg : $errorMsg";
  }
}

enum BdliveForceOfflineReason {
  // 在开启禁止重复登录功能后，同一观众在多端同时登录同一直播间
  multiLogin,
  // 超时未签到
  checkInTimeout,
}

// 清晰度
enum BdliveVideoResolution {
  unknown, // 未知
  ld, // 流畅
  sd, // 标清
  hd, // 高清
  uhd, // 超清
  bd, // 蓝光
  origin, // 原画
  auto, // 智能
}

