//
//  bdlive_flutter_viewer_api.dart
//
//  bdlive-flutter-viewer SDK License
//
//  Copyright 2025 Byteplus Pte. Ltd. All Rights Reserved.
//
//  The bdlive-flutter-viewer SDK was developed by Byteplus Pte. Ltd. (hereinafter “Byteplus”).
//  Any copyright or patent right is owned by and proprietary material of the Byteplus.
//
//  bdlive-flutter-viewer SDK is available under the Byteplus product and licensed under the commercial license.
//  Customers can contact via https://www.byteplus.com/en/contact for commercial licensing options.
//  Here is also a link to Service Level Agreement: https://docs.byteplus.com/en/byteplus-livesaas/docs/byteplus-live-service-level-agreement
//
//  Without Byteplus's prior written permission, any use of bdlive-flutter-viewer SDK, in particular any use for commercial purposes, is prohibited.
//  This includes, without limitation, incorporation in a commercial product, use in a commercial service, or production of other artefacts for commercial purposes.
//
//  Without Byteplus's prior written permission, the bdlive-flutter-viewer SDK may not be reproduced, modified and/or made available in any form to any third party.
import '../src/bdlive_flutter_viewer_impl.dart';
import 'bdlive_model.dart';
import 'config/bdlive_viewer_config.dart';

class BdlLiveRoomStatusListener {
  /**
   * Callback when entering the live room page foreground
   * @param activity: Configuration when entering the live room, see BdlActivityConfig
   */
  void Function(BdlActivityConfig activity)? onLiveRoomActivityResume;

  /**
   * Callback when entering the live room page background
   * @param activity: Configuration when entering the live room, see BdlActivityConfig
   */
  void Function(BdlActivityConfig activity)? onLiveRoomActivityPause;

  /**
   * Callback when entering the live room page destoryed
   * @param activity: Configuration when entering the live room, see BdlActivityConfig
   */
  void Function(BdlActivityConfig activity)? onLiveRoomFullDestroyed;

  /**
   * Callback when the floating window of the live room is about to appear
   * @param activity: Configuration when entering the live room, see BdlActivityConfig
   */
  void Function(BdlActivityConfig activity)? onFloatViewWillAppear;

  /**
   * Callback when the floating window of the live room is about to close
   * @param activity: Configuration when entering the live room, see BdlActivityConfig
   */
  void Function(BdlActivityConfig activity)? onFloatViewWillClose;

  /**
   * Callback when an ad/product card in the live room is clicked, allowing the client to listen to this callback for custom redirection
   * @param url: The redirection link corresponding to the ad/product card
   * @param type: The type of ad/product card, see BdlViewerOpenUrlType
   * @param enableFloating: Whether the user has set to display content as a floating layer in the corporate live broadcast console.
      true: Display content as a floating layer on the viewing page.
      false: Display content on a new page.
   */
  void Function(String url, BdlViewerOpenUrlType type, bool enableFloating)?
      onOpenUrlCallback;

  /**
   * Callback triggered when join live room logically success or user clicked pip. Listen to this callback to show the live pull view.
   * @param activity: Configuration when entering the live room, see BdlActivityConfig
   */
  void Function(BdlActivityConfig activity)? showLivePullView;

  /**
   * Callback triggered when the close button in the live room is clicked. Listen to this callback to hide the live pull view.
   * @param activity: Configuration when entering the live room, see BdlActivityConfig
   */
  void Function(BdlActivityConfig activity)? hideLivePullView;

  BdlLiveRoomStatusListener({
    this.onLiveRoomActivityResume,
    this.onLiveRoomActivityPause,
    this.onLiveRoomFullDestroyed,
    this.onFloatViewWillAppear,
    this.onFloatViewWillClose,
    this.onOpenUrlCallback,
    this.showLivePullView,
    this.hideLivePullView,
  });
}

abstract class BdliveViewerApi {
  static BdliveViewerApi _instance = BdliveFlutterViewerImpl();

  static BdliveViewerApi get instance => _instance;

  static set instance(BdliveViewerApi instance) {
    _instance = instance;
  }

  /**
   * Set the configuration for the complete live room page
   * @param config: Configuration for the complete live room page, see BdLiveViewerConfig
   */
  Future<void> setLiveRoomConfig(BdLiveViewerConfig config);

  /**
   * Get the configuration for the complete live room page
   * @return: Configuration for the complete live room page, see BdLiveViewerConfig
   */
  BdLiveViewerConfig getLiveRoomConfig();

  /**
   * Enter the complete live room page logicly, not show live room page
   * @param activity: Configuration when entering the live room, see BdlActivityConfig
   */
  Future<bool> enterLiveRoomLogically(BdlActivityConfig activity);

  /**
   * Leave the complete live room page. If the floating window of the live room is currently displayed, it will also be destroyed.
   */
  Future<void> leaveLiveRoomLogically();

  /**
   * Enter the complete live room page
   * @param activity: Configuration when entering the live room, see BdlActivityConfig
   */
  Future<bool> enterLiveRoom(BdlActivityConfig activity);

  // /**
  //  * Leave the complete live room page. If the floating window of the live room is currently displayed, it will also be destroyed.
  //  */
  // Future<void> leaveLiveRoomLogically();

  /**
   * Enter the complete live room page
   * @param activity: Configuration when entering the live room, see BdlActivityConfig
   */
  // Future<bool> enterLiveRoom(BdlActivityConfig activity);

  /**
   * Leave the complete live room page. If the floating window of the live room is currently displayed, it will also be destroyed.
   */
  Future<void> leaveLiveRoom();

  /**
   * Simulate manually clicking the exit live room icon in the complete live room page; subsequent behavior is the same as that of a viewer clicking the exit live room icon.
   */
  Future<void> hideLiveRoom();

  /**
   * Set a listener for the complete live room page to listen for events of the complete live room page
   * @param listener: Listener for the complete live room page, see BdlLiveRoomStatusListener
   */
  Future<void> setLiveRoomStatusListener(BdlLiveRoomStatusListener? listener);

  /**
   * Used in conjunction with the onOpenUrlCallback callback in BdlLiveRoomStatusListener,
   * clients can call this API to use some default redirection capabilities of the SDK
   * @param url: The redirection link corresponding to the ad/product card
   * @param type: The type of ad/product card, see BdlViewerOpenUrlType
   * @param enableFloating: Whether to allow displaying content as a floating layer on the viewing page.
      true: Display content as a floating layer on the viewing page; a webview pop-up will be shown in the complete live page to display the URL.
      false: Display content on a new page; the app's deep link mechanism will be used to find and open the corresponding page.
   */
  Future<void> openUrl(
      String url, BdlViewerOpenUrlType type, bool enableFloating);

  /**
   * Close the floating window of the live room
   */
  Future<void> closeFloatingView();

  Future<void> dispatchWidgetSate(int state);

  Future<bool> onWillClose();
}
