//
//  BdliveFlutterViewerPlayerDelegate.m
//
//  bdlive-flutter-viewer SDK License
//
//  Copyright 2025 Byteplus Pte. Ltd. All Rights Reserved.
//
//  The bdlive-flutter-viewer SDK was developed by Byteplus Pte. Ltd. (hereinafter “Byteplus”).
//  Any copyright or patent right is owned by and proprietary material of the Byteplus.
//
//  bdlive-flutter-viewer SDK is available under the Byteplus product and licensed under the commercial license.
//  Customers can contact via https://www.byteplus.com/en/contact for commercial licensing options.
//  Here is also a link to Service Level Agreement: https://docs.byteplus.com/en/byteplus-livesaas/docs/byteplus-live-service-level-agreement
//
//  Without Byteplus's prior written permission, any use of bdlive-flutter-viewer SDK, in particular any use for commercial purposes, is prohibited.
//  This includes, without limitation, incorporation in a commercial product, use in a commercial service, or production of other artefacts for commercial purposes.
//
//  Without Byteplus's prior written permission, the bdlive-flutter-viewer SDK may not be reproduced, modified and/or made available in any form to any third party.

#import "BdliveFlutterViewerPlayerDelegate.h"
#import "NSError+BdliveToMap.h"
#import <BPLive/BPLive.h>

#define BDL_FLUTTER_FLT_EPSILON  0.000001

@interface BdliveFlutterViewerPlayerDelegate () <BDLBasicService>

@property (nonatomic, assign) CGFloat vodDuration;
@property (nonatomic, assign) BOOL isPlaying;
@property (nonatomic, assign) BOOL isPlayError;
@property (nonatomic, strong) NSTimer *timer;
@property (nonatomic, assign) int prevPlayTimeInMs;

@end

@implementation BdliveFlutterViewerPlayerDelegate

- (void)notifyLiveRoomStatusChange {
    [self activityStatusDidChange:[[BDLLiveEngine sharedInstance] getActivityService].getActivityModel.basic.status];
}

- (void)activityStatusDidChange:(BDLActivityStatus)status {
    [self emitEvent:@{
        @"status" : @(status),
    } methodName:@"liveRoomStatusChanged"];
}

- (void)notifyPlayableStatusChange:(BdliveFlutterViewerPlayerPlayableStatus)playableStatus {
    [self emitEvent:@{
        @"playableStatus" : @(playableStatus)
    } methodName:@"playableStatusChanged"];
}

- (void)notifyVodPlayTimeChange:(int)curTimeInMills {
    [self emitEvent:@{
        @"curTimeInMills" : @(curTimeInMills)
    } methodName:@"vodCurPlayTimeChanged"];
}

- (void)setIsPlaying:(BOOL)isPlaying {
    if (_isPlaying == isPlaying) {
        return;
    }
    _isPlaying = isPlaying;
    [self emitEvent:@{
        @"isPlaying" : @(isPlaying)
    } methodName:@"isPlayingChanged"];
}

- (void)basePlayerView:(BDLBasePlayerView *)basePlayerView videoSizeDidChange:(CGSize)videoSize {
    [self emitEvent:@{
        @"width" : @((int)(videoSize.width)),
        @"height" : @((int)(videoSize.height)),
    } methodName:@"videoSizeChanged"];
}

- (void)basePlayerViewStallStart:(BDLBasePlayerView *)basePlayerView {
    [self emitEvent:@{
        @"isStalling" : @(YES),
    } methodName:@"isStallingChanged"];
}

- (void)basePlayerViewStallEnd:(BDLBasePlayerView *)basePlayerView {
    [self emitEvent:@{
        @"isStalling" : @(NO),
    } methodName:@"isStallingChanged"];
}

- (void)basePlayerView:(BDLBasePlayerView *)basePlayerView vodDidFinishWithError:(NSError *)error {
    NSDictionary *errMap = [error bdlive_toMap];
    if (errMap) {
        [self emitEvent:@{
            @"error" : errMap
        } methodName:@"vodErrorOccurred"];
    }
}

- (void)setIsPlayError:(BOOL)isPlayError {
    if (_isPlayError == isPlayError) {
        return;
    }
    _isPlayError = isPlayError;
    
    [self emitEvent:@{
        @"isPlayError" : @(isPlayError)
    } methodName:@"playErrorStatusChanged"];
}

- (void)basePlayerView:(BDLBasePlayerView *)basePlayerView vodLoadStateDidChange:(TTVideoEngineLoadState)loadState {
    if (loadState == TTVideoEngineLoadStatePlayable) {
        [self emitEvent:nil methodName:@"vodPrepared"];
        
        if (fabs(basePlayerView.duration - self.vodDuration) > BDL_FLUTTER_FLT_EPSILON) {
            [self emitEvent:@{
                @"durationInMills" : @((int)(basePlayerView.duration * 1000)),
            } methodName:@"vodDurationChanged"];
            self.vodDuration = basePlayerView.duration;
        }
    }
}

- (void)stopTimerIfNeeded {
    if (self.timer) {
        [self.timer invalidate];
        self.timer = nil;
    }
}

- (void)basePlayerView:(BDLBasePlayerView *)basePlayerView vodPlaybackStateDidChange:(TTVideoEnginePlaybackState)playbackState {
    self.isPlaying = playbackState == TTVideoEnginePlaybackStatePlaying;
    if (playbackState == TTVideoEnginePlaybackStatePlaying) {
        __weak typeof(self) weakSelf = self;
        __weak typeof(basePlayerView) weakPlayer = basePlayerView;
        [self stopTimerIfNeeded];
        self.timer = [NSTimer timerWithTimeInterval:1 repeats:YES block:^(NSTimer * _Nonnull timer) {
            if (!weakPlayer.isVod) {
                [weakSelf stopTimerIfNeeded];
                return;
            }
            int playTime = (int)(weakPlayer.currentPlaybackTime * 1000);
            if (playTime == weakSelf.prevPlayTimeInMs) {
                return;
            }
            weakSelf.prevPlayTimeInMs = playTime;
            [weakSelf notifyVodPlayTimeChange:playTime];
        }];
        [[NSRunLoop currentRunLoop] addTimer:self.timer forMode:NSRunLoopCommonModes];
    }
    else {
        [self stopTimerIfNeeded];
    }
}

- (void)basePlayerView:(BDLBasePlayerView *)basePlayerView onVodAutoContinuePlayback:(NSTimeInterval)playbackTime {
    [self emitEvent:@{
        @"seekTimeInMills" : @((int)(playbackTime * 1000)),
    } methodName:@"vodAutoSeekPreviousTime"];
}

- (void)basePlayerView:(BDLBasePlayerView *)basePlayerView liveLoadStateDidChange:(BDLPlayerLoadState)loadState {
    if (loadState == BDLPlayerLoadStatePlayable) {
        [self emitEvent:nil methodName:@"livePrepared"];
    }
    else if (loadState == BDLPlayerLoadStateError) {
        NSDictionary *errorMap = [basePlayerView.liveError bdlive_toMap];
        if (errorMap) {
            [self emitEvent:@{
                @"error" : errorMap
            } methodName:@"liveErrorOccurred"];
        }
    }
}

- (void)basePlayerView:(BDLBasePlayerView *)basePlayerView livePlaybackStateDidChange:(BDLPlayerState)playbackState {
    self.isPlaying = playbackState == BDLPlayerStatePlaying;
}

- (void)basePlayerViewOnFirstVideoFrameRendered:(BDLBasePlayerView *)basePlayerView {
    if (basePlayerView.isLive) {
        [self emitEvent:@{
            @"isFirstFrame" : @(YES)
        } methodName:@"liveFirstFrameRendered"];
    }
    else {
        [self emitEvent:nil methodName:@"vodRenderStarted"];
    }
}

- (void)basePlayerView:(BDLBasePlayerView *)basePlayerView coverImageViewDidAppear:(UIImageView *)coverImageView {
    [self emitEvent:@{
        @"isVisible" : @(YES)
    } methodName:@"coverImageVisibleChanged"];
}

- (void)basePlayerView:(BDLBasePlayerView *)basePlayerView coverImageViewDidDisappear:(UIImageView *)coverImageView {
    [self emitEvent:@{
        @"isVisible" : @(NO)
    } methodName:@"coverImageVisibleChanged"];
}

- (void)basePlayerView:(BDLBasePlayerView *)basePlayerView supportedVideoResolutionsDidChange:(NSArray<NSNumber *> *)supportedVideoResolutions currentVideoResolution:(BDLVideoResolution)currentVideoResolution {
    if (!supportedVideoResolutions || !currentVideoResolution) {
        return;
    }
    [self emitEvent:@{
        @"resolutions" : supportedVideoResolutions,
        @"defaultResolution" : @(currentVideoResolution),
    } methodName:@"resolutionInfoChanged"];
}

- (void)basePlayerView:(BDLBasePlayerView *)basePlayerView coverImageDidChange:(UIImage *)image error:(NSError *)error {
    if (!image) {
        return;
    }
    [self emitEvent:@{
        @"width" : @((int)(image.size.width)),
        @"height" : @((int)(image.size.height))
    } methodName:@"coverImageSizeChanged"];
}

- (void)basePlayerView:(BDLBasePlayerView *)basePlayerView onUserForceOffline:(BDLForceOfflineReason)reason {
    [self emitEvent:@{
        @"reason" : @(reason)
    } methodName:@"onUserForceOffline"];
}

- (void)destroy {
    [self emitEvent:nil methodName:@"onRelease"];
    [super destroy];
}

@end
