//
//  BdliveFlutterViewerPlayer.m
//
//  bdlive-flutter-viewer SDK License
//
//  Copyright 2025 Byteplus Pte. Ltd. All Rights Reserved.
//
//  The bdlive-flutter-viewer SDK was developed by Byteplus Pte. Ltd. (hereinafter “Byteplus”).
//  Any copyright or patent right is owned by and proprietary material of the Byteplus.
//
//  bdlive-flutter-viewer SDK is available under the Byteplus product and licensed under the commercial license.
//  Customers can contact via https://www.byteplus.com/en/contact for commercial licensing options.
//  Here is also a link to Service Level Agreement: https://docs.byteplus.com/en/byteplus-livesaas/docs/byteplus-live-service-level-agreement
//
//  Without Byteplus's prior written permission, any use of bdlive-flutter-viewer SDK, in particular any use for commercial purposes, is prohibited.
//  This includes, without limitation, incorporation in a commercial product, use in a commercial service, or production of other artefacts for commercial purposes.
//
//  Without Byteplus's prior written permission, the bdlive-flutter-viewer SDK may not be reproduced, modified and/or made available in any form to any third party.

#import "BdliveFlutterViewerPlayer.h"
#import "BdliveFlutterStreamHandler.h"
#import "BdliveFlutterViewerPlayerDelegate.h"
#import "BdliveFlutterViewerPlugin.h"

#import <Masonry/Masonry.h>
#import <BPLive/BPLive.h>

@interface BdliveFlutterViewerPlayer () <BDLBasicService>

@property (nonatomic, weak) NSObject<FlutterPluginRegistrar> *registrar;
@property (nonatomic, assign) int64_t insId;
@property (nonatomic, strong) BdliveFlutterMethodHandler *methodHandler;
@property (nonatomic, strong) UIView *containerView;
@property (nonatomic, strong) BDLBasePlayerView *playerView;

@property (nonatomic, strong) BdliveFlutterViewerPlayerDelegate *playerDelegate;

@property (nonatomic, assign) BdliveFlutterViewerPlayerPlayableStatus playableStatus;

@end

@implementation BdliveFlutterViewerPlayer

- (UIView *)view {
    return self.containerView;
}

- (instancetype)initWithRegistrar:(NSObject<FlutterPluginRegistrar> *)registrar insId:(int64_t)insId {
    self = [super init];
    if (self) {
        self.registrar = registrar;
        self.insId = insId;
        
        self.methodHandler = [BdliveFlutterMethodHandler methodChannelWithName:[NSString stringWithFormat:@"com.bytedance.live_player.%lld", self.insId]
                                                                   methodTarget:self
                                                                binaryMessenger:[self.registrar messenger]];
        self.containerView = [[UIView alloc] init];
        
        [[BdliveFlutterViewerPlugin sharedInstance].viewerViewFactory addPlayer:self instanceId:insId];
        [[BDLLiveEngine sharedInstance] registerService:self];
    }
    return self;
}

- (void)activityBasicDidChange:(BDLBasicModel *)model {
    [self updatePlayableStatus:model];
}

- (void)destroy:(NSDictionary *)arguments result:(FlutterResult)result {
    [self.methodHandler destroy];
    self.methodHandler = nil;
    
    [self.playerDelegate destroy];
    self.playerDelegate = nil;
    
    [self.playerView stop];
    [self.playerView removeFromSuperview];
    self.playerView = nil;
    
    if (self.destroyBlock) {
        self.destroyBlock(self);
    }
    result(nil);
}

- (void)start:(NSDictionary *)arguments result:(FlutterResult)result {
    if (!self.activity || !self.activity.isValid) {
        result(nil);
        return;
    }
    BDLLiveEngine *engine = [BDLLiveEngine sharedInstance];
    [engine liveRoomConfiguration].playerConfig.common.enableLivePiP = NO;
    [engine liveRoomConfiguration].playerConfig.common.enableVodPiP = NO;
    
    [engine joinLiveRoomWithActivity:self.activity success:^{
        self.playerView = [[BDLBasePlayerView alloc] init];
        self.playerView.enablePictureInPicture = NO;
        [self.containerView addSubview:self.playerView];
        if (self.playerDelegate) {
            self.playerView.delegate = self.playerDelegate;
            [[BDLServiceLocator sharedInstance] registerService:self.playerDelegate];
            [self.playerDelegate notifyLiveRoomStatusChange];
        }
        [self.playerView mas_makeConstraints:^(MASConstraintMaker *make) {
            make.edges.equalTo(self.containerView);
        }];
        
        [self updatePlayableStatus:engine.getActivityService.getActivityModel.basic];
        result(nil);
    } failure:^(NSError * _Nonnull error) {
        result(nil);
    }];
}

- (void)updatePlayableStatus:(BDLBasicModel *)basic {
    BdliveFlutterViewerPlayerPlayableStatus status = BdliveFlutterViewerPlayerPlayableStatusUnknown;
    if (basic.status == BDLActivityStatusLive) {
        status = BdliveFlutterViewerPlayerPlayableStatusLiveCanPlay;
    }
    else if (basic.status == BDLActivityStatusReplay
             || (basic.status == BDLActivityStatusPreview && basic.previewVideoUrl.length > 0)) {
        status = BdliveFlutterViewerPlayerPlayableStatusVodCanPlay;
    }
    self.playableStatus = status;
}

- (void)play:(NSDictionary *)arguments result:(FlutterResult)result {
    if (self.playableStatus != BdliveFlutterViewerPlayerPlayableStatusUnknown) {
        [self.playerView play];
    }
    result(nil);
}

- (void)pause:(NSDictionary *)arguments result:(FlutterResult)result {
    if (self.playableStatus != BdliveFlutterViewerPlayerPlayableStatusUnknown) {
        [self.playerView pause];
    }
    result(nil);
}

- (void)setDelegate:(NSDictionary *)arguments result:(FlutterResult)result {
    if (self.playerDelegate) {
        [self.playerDelegate destroy];
    }
    BdliveFlutterViewerPlayerDelegate *delegate = [BdliveFlutterViewerPlayerDelegate eventHandlerWithName:[NSString stringWithFormat:@"com.bytedance.live_player.delegate.event.%lld", self.insId]
                                                                                        binaryMessenger:[self.registrar messenger]];
    
    self.playerDelegate = delegate;
    if (self.playerView) {
        [self.playerView setDelegate:delegate];
        [[BDLLiveEngine sharedInstance] registerService:self.playerDelegate];
        [self.playerDelegate notifyLiveRoomStatusChange];
    }
    
    result(nil);
}

- (void)refreshLive:(NSDictionary *)arguments result:(FlutterResult)result {
    if (self.playableStatus != BdliveFlutterViewerPlayerPlayableStatusLiveCanPlay) {
        result(nil);
        return;
    }
    [self.playerView pause];
    [self.playerView play];
    result(nil);
}

- (void)isPlaying:(NSDictionary *)arguments result:(FlutterResult)result {
    BOOL isPlaying = NO;
    if (self.playableStatus != BdliveFlutterViewerPlayerPlayableStatusUnknown) {
        isPlaying = self.playerView.isPlaying;
    }
    result(@(isPlaying));
}

- (void)isStalling:(NSDictionary *)arguments result:(FlutterResult)result {
    BOOL isStalling = NO;
    if (self.playableStatus != BdliveFlutterViewerPlayerPlayableStatusUnknown) {
        isStalling = self.playerView.isStalling;
    }
    result(@(isStalling));
}

- (void)getPlayableStatus:(NSDictionary *)arguments result:(FlutterResult)result {
    result(@(self.playableStatus));
}

- (void)setPlayerLayoutMode:(NSDictionary *)arguments result:(FlutterResult)result {
    NSInteger mode = [arguments[@"playerLayoutMode"] integerValue];
    id value = [[self flutterModeToBDLPlayerScalingModeDict] objectForKey:@(mode)];
    if (value) {
        [self.playerView setScalingMode:[value integerValue]];
    }
    result(nil);
}

- (void)getPlayerLayoutMode:(NSDictionary *)arguments result:(FlutterResult)result {
    BDLPlayerScalingMode mode = self.playerView.scalingMode;
    NSNumber *flutterValue = nil;
    NSDictionary *dict = [self flutterModeToBDLPlayerScalingModeDict];
    for (NSNumber *key in dict.allKeys) {
        if ([[dict objectForKey:key] integerValue] == mode) {
            flutterValue = key;
            break;
        }
    }
    result(flutterValue);
}

- (NSDictionary *)flutterModeToBDLPlayerScalingModeDict {
    return @{
        @0 : @(BDLPlayerScalingModeAspectFit),
        @1 : @(BDLPlayerScalingModeFill),
        @2 : @(BDLPlayerScalingModeAspectFill),
    };
}

- (void)getResolutions:(NSDictionary *)arguments result:(FlutterResult)result {
    NSArray *resolutions = nil;
    if (self.playableStatus != BdliveFlutterViewerPlayerPlayableStatusUnknown) {
        resolutions = self.playerView.supportedVideoResolutions;
    }
    result(resolutions);
}

- (void)getCurResolution:(NSDictionary *)arguments result:(FlutterResult)result {
    NSNumber *resolution = nil;
    if (self.playableStatus != BdliveFlutterViewerPlayerPlayableStatusUnknown) {
        resolution = @(self.playerView.currentVideoResolution);
    }
    result(resolution);
}

- (void)setCurResolution:(NSDictionary *)arguments result:(FlutterResult)result {
    if (self.playableStatus != BdliveFlutterViewerPlayerPlayableStatusUnknown) {
        NSInteger resolutionValue = [arguments[@"resolution"] integerValue];
        [self.playerView setCurrentVideoResolution:resolutionValue];
    }
    
    result(nil);
}

- (void)getCurVodPlayTime:(NSDictionary *)arguments result:(FlutterResult)result {
    CGFloat time = 0;
    if (self.playableStatus != BdliveFlutterViewerPlayerPlayableStatusUnknown) {
        time = self.playerView.currentPlaybackTime;
    }
    result(@((int)(time * 1000)));
}


- (void)getCurVodDuration:(NSDictionary *)arguments result:(FlutterResult)result {
    CGFloat duration = 0;
    if (self.playableStatus != BdliveFlutterViewerPlayerPlayableStatusUnknown) {
        duration = self.playerView.duration;
    }
    result(@((int)(duration * 1000)));
}

- (void)seekVodTime:(NSDictionary *)arguments result:(FlutterResult)result {
    if (self.playableStatus == BdliveFlutterViewerPlayerPlayableStatusUnknown) {
        result(nil);
    }
    [self.playerView seek:[[arguments objectForKey:@"timeInMills"] integerValue] / 1000.0 completion:^(BOOL success) {
        result(nil);
    }];
}

- (void)setMute:(NSDictionary *)arguments result:(FlutterResult)result {
    self.playerView.muted = [arguments[@"isMute"] boolValue];
    result(nil);
}

- (void)setVodLoop:(NSDictionary *)arguments result:(FlutterResult)result {
    self.playerView.looping = [arguments[@"isLoop"] boolValue];
    result(nil);
}

- (void)setVodPlaySpeed:(NSDictionary *)arguments result:(FlutterResult)result {
    self.playerView.playbackSpeed = [arguments[@"playSpeed"] floatValue];
    result(nil);
}

- (void)getCurVodPlaySpeed:(NSDictionary *)arguments result:(FlutterResult)result {
    CGFloat speed = -1;
    if (self.playableStatus != BdliveFlutterViewerPlayerPlayableStatusUnknown) {
        speed = self.playerView.playbackSpeed;
    }
    result(@(speed));
}

- (void)setInBackground:(NSDictionary *)arguments result:(FlutterResult)result {
    self.playerView.visible = ![arguments[@"inBackground"] boolValue];
    result(nil);
}

- (void)getLiveRoomStatus:(NSDictionary *)arguments result:(FlutterResult)result {
    BDLActivityStatus status = BDLActivityStatusUnknown;
    if (self.playableStatus != BdliveFlutterViewerPlayerPlayableStatusUnknown) {
        status = [[BDLLiveEngine sharedInstance].getActivityService getActivityModel].basic.status;
    }
    result(@(status));
}

//设置是否强制隐藏封面图
- (void)setCoverImageContainerVisibility:(NSDictionary *)arguments result:(FlutterResult)result {
    self.playerView.coverImageView.alpha = [arguments[@"visible"] boolValue] ? 1 : 0;
    result(nil);
}

- (void)setPlayableStatus:(BdliveFlutterViewerPlayerPlayableStatus)playableStatus {
    if (_playableStatus == playableStatus) {
        return;
    }
    _playableStatus = playableStatus;
    [self.playerDelegate notifyPlayableStatusChange:playableStatus];
}

@end
