//
//  BdliveFlutterViewerPlugin.m
//
//  bdlive-flutter-viewer SDK License
//
//  Copyright 2025 Byteplus Pte. Ltd. All Rights Reserved.
//
//  The bdlive-flutter-viewer SDK was developed by Byteplus Pte. Ltd. (hereinafter “Byteplus”).
//  Any copyright or patent right is owned by and proprietary material of the Byteplus.
//
//  bdlive-flutter-viewer SDK is available under the Byteplus product and licensed under the commercial license.
//  Customers can contact via https://www.byteplus.com/en/contact for commercial licensing options.
//  Here is also a link to Service Level Agreement: https://docs.byteplus.com/en/byteplus-livesaas/docs/byteplus-live-service-level-agreement
//
//  Without Byteplus's prior written permission, any use of bdlive-flutter-viewer SDK, in particular any use for commercial purposes, is prohibited.
//  This includes, without limitation, incorporation in a commercial product, use in a commercial service, or production of other artefacts for commercial purposes.
//
//  Without Byteplus's prior written permission, the bdlive-flutter-viewer SDK may not be reproduced, modified and/or made available in any form to any third party.
#import "BdliveFlutterViewerPlugin.h"
#import <objc/message.h>
#import <TTSDKFramework/TTSDKFramework.h>
#import <BPLive/BPLive.h>
#import "BdliveFlutterViewerDelegate.h"
#import "BdliveFlutterViewerPlayer.h"
#import "BdliveFlutterConfig.h"
#import "BdliveLivePullView.h"
#import "BDLLivePullViewController+Flutter.h"

@interface BDLEnv ()

+ (void)setChannelType:(BDLChannelType)type;

@end

@interface BdliveFlutterViewerPlugin ()

@property (nonatomic, weak) NSObject<FlutterPluginRegistrar> *registrar;
@property (nonatomic, strong) BdliveFlutterViewerDelegate *delegate;
@property (nonatomic, strong) NSMutableDictionary *playerMap;

@property (nonatomic, strong) BDLActivity *activity;

@end

@implementation BdliveFlutterViewerPlugin

+ (instancetype)sharedInstance {
    static BdliveFlutterViewerPlugin *instance = nil;
    static dispatch_once_t onceToken;
    dispatch_once(&onceToken, ^{
        instance = [[self alloc] init];
        if ([BDLEnv respondsToSelector:@selector(setChannelType:)]) {
            [BDLEnv setChannelType:BDLChannelTypeFlutter];
        }
    });
    return instance;
}

+ (void)registerWithRegistrar:(NSObject<FlutterPluginRegistrar>*)registrar {
    BdliveFlutterViewerPlugin *plugin = [self sharedInstance];
    plugin.registrar = registrar;
    
    FlutterMethodChannel *channel = [FlutterMethodChannel
                                     methodChannelWithName:@"com.bytedance.flutter_viewer.plugin"
                                     binaryMessenger:[registrar messenger]];
    [registrar addMethodCallDelegate:plugin channel:channel];
    
    BdliveViewerViewFactory *fc = [[BdliveViewerViewFactory alloc] init];
    [registrar registerViewFactory:fc withId:@"BdliveViewerPlayerView"];
    plugin.viewerViewFactory = fc;
    BdliveLivePullViewFactory *factory = [[BdliveLivePullViewFactory alloc] initWithMessenger:registrar.messenger];
    [registrar registerViewFactory:factory withId:@"BdlLivePullView"];
}

- (void)handleMethodCall:(FlutterMethodCall*)call result:(FlutterResult)result {
    SEL selector = NSSelectorFromString([call.method stringByAppendingString:@":result:"]);
    if ([self respondsToSelector:selector]) {
        ((void(*)(id,SEL,id,FlutterResult))objc_msgSend)(self, selector, call.arguments, result);
        return;
    }
    else {
        result(FlutterMethodNotImplemented);
    }
}

- (void)startWithConfiguration:(NSDictionary *)arguments result:(FlutterResult)result {
    NSString *appID = arguments[@"appID"];
    NSString *bundleID = arguments[@"bundleID"];
    NSString *appName = arguments[@"appName"];
    NSString *channel = arguments[@"channel"];
    NSString *appVersion = arguments[@"appVersion"];
    NSString *liveLicenseFilePath = arguments[@"liveLicenseFilePath"];
    NSString *vodLicenseFilePath = arguments[@"vodLicenseFilePath"];
    
    TTSDKConfiguration *configuration = [TTSDKConfiguration defaultConfigurationWithAppID:appID];
    if (bundleID) {
        configuration.bundleID = bundleID;
    }
    if (appName) {
        configuration.appName = appName;
    }
    if (channel) {
        configuration.channel = channel;
    }
    if (appVersion) {
        configuration.appVersion = appVersion;
    }
    
    NSString *key = [self.registrar lookupKeyForAsset:vodLicenseFilePath];
    NSString *path = [[NSBundle mainBundle] pathForResource:key ofType:nil];
    configuration.bizType = TTSDKServiceBizType_All & ~TTSDKServiceBizType_Live;
    configuration.licenseFilePath = path;
    [TTSDKManager startWithConfiguration:configuration];
    
    key = [self.registrar lookupKeyForAsset:liveLicenseFilePath];
    path = [[NSBundle mainBundle] pathForResource:key ofType:nil];
    configuration.bizType = TTSDKServiceBizType_Live;
    configuration.licenseFilePath = path;
    [TTSDKManager startWithConfiguration:configuration];
    
    result(nil);
}

- (void)enterLiveRoomLogically:(NSDictionary *)arguments result:(FlutterResult)result {
    [self joinLiveRoom:arguments autoShowView:NO result:result];
}

- (void)enterLiveRoom:(NSDictionary *)arguments result:(FlutterResult)result {
    [self joinLiveRoom:arguments autoShowView:YES result:result];
}

- (void)joinLiveRoom:(NSDictionary *)arguments autoShowView:(BOOL)autoShowView result:(FlutterResult)result {
    BDLActivity *activity = [[BDLActivity alloc] init];
    
    activity.activityId = arguments[@"activityId"];
    activity.token = arguments[@"token"];
    activity.authMode = [arguments[@"authMode"] integerValue];
    activity.isPortrait = [arguments[@"isPortrait"] boolValue];
    
    BDLLiveEngine *engine = [BDLLiveEngine sharedInstance];
    BDLLivePullViewController *livePullVC = [engine getLivePullViewController];
    if ([self.activity isEqual:activity] && livePullVC) { // No more authentication is required. The live room is displayed directly.
        if (livePullVC.isFloating) {
            [livePullVC hideFloatingPlayerIfAvailable:YES];
        }
        else {
            [self configLivePullVC:livePullVC autoShowView:autoShowView];
            [self.delegate showLivePullViewController:livePullVC];
        }
        result(@(YES));
        return;
    }
    
    [engine joinLiveRoomWithActivity:activity success:^{
        self.activity = activity;
        
        BDLLivePullViewController *livePullVC = [engine getLivePullViewController];
        [self configLivePullVC:livePullVC autoShowView:autoShowView];
        [self.delegate showLivePullViewController:livePullVC];
        
        result(@(YES));
    } failure:^(NSError * _Nonnull error) {
        result(@(NO));
    }];
}

- (void)configLivePullVC:(BDLLivePullViewController *)livePullVC autoShowView:(BOOL)autoShowView {
    self.delegate.autoShowView = autoShowView;
    self.delegate.activity = self.activity;
    [livePullVC setDelegate:self.delegate];
    livePullVC.actionProvider = self.delegate;
    
    __weak typeof(self) weakSelf = self;
    livePullVC.config.customShoppingCardController = ^(BDLLivePullViewController * _Nonnull viewController, BDLShoppingCardController * _Nonnull shoppingCardController) {
        shoppingCardController.onCardClickBlock = ^(BDLShoppingCardController * _Nonnull controller, NSURL * _Nonnull url, BOOL isFloatingEnable, BDLCardModel * _Nonnull cardModel, BDLShoppingCardClickPosition clickPosition) {
            __strong typeof(weakSelf) self = weakSelf;
            [self.delegate openUrl:[url absoluteString] enableFloating:isFloatingEnable type:BDLPageAdvertisementTypeCardAd];
        };
    };

    livePullVC.config.customizePageAdView = ^UIView<BDLPageAdViewProtocol> * _Nullable(BDLLivePullViewController * _Nonnull viewController, BDLPageAdView * _Nonnull view) {
        view.viewTappedBlock = ^(BDLPageAdView * _Nonnull view, BDLAdvertisementModel * _Nonnull ad, BOOL isFloatingEnable) {
            __strong typeof(weakSelf) self = weakSelf;
            [self.delegate openUrl:ad.advertisementRedirectUrl enableFloating:isFloatingEnable type:ad.pageAdvertisementType];
        };
        return view;
    };
    
    livePullVC.config.customFloatingController = ^(BDLLivePullViewController * _Nonnull viewController, BDLFloatingController * _Nonnull floatingController) {
        floatingController.openUrlBlock = ^(BDLFloatingController * _Nonnull floatingController, NSURL * _Nonnull url, BOOL isFloatingEnable) {
            __strong typeof(weakSelf) self = weakSelf;
            [self.delegate openUrl:[url absoluteString] enableFloating:isFloatingEnable type:BDLPageAdvertisementTypeFloatingAd];
        };
    };
    livePullVC.floatingPlayerViewWillAppear = ^(BDLLivePullViewController * _Nonnull viewController, BDLFloatingPlayer * _Nonnull floatingPlayer) {
        __strong typeof(weakSelf) self = weakSelf;
        [self.delegate emitEvent:@{
            @"activity" : [self.activity yy_modelToJSONObject]
        } methodName:@"onFloatViewWillAppear"];
    };
    livePullVC.floatingPlayerViewWillClose = ^(BDLLivePullViewController * _Nonnull viewController, BDLFloatingPlayer * _Nonnull floatingPlayer) {
        __strong typeof(weakSelf) self = weakSelf;
        [self.delegate emitEvent:@{
            @"activity" : [self.activity yy_modelToJSONObject]
        } methodName:@"onFloatViewWillClose"];
    };
    livePullVC.onFloatingPlayerCloseTapped = ^BOOL(BDLLivePullViewController * _Nonnull viewController, BDLFloatingPlayer * _Nonnull floatingPlayer) {
        if (viewController.isFlutterViewAlive) {
            [viewController hideFloatingPlayerIfAvailable:NO];
            return NO;
        }
        else {
            return YES;
        }
    };
    livePullVC.config.shouldShowInAppPipIfAvailable = ^BOOL(BDLLivePullViewController * _Nonnull viewController, BDLActivityStatus status, BOOL isClose) {
        if (!BdliveFlutterConfig.sharedInstance.playerConfig.enablePIP) {
            return NO;
        }
        if (isClose) {
            return BdliveFlutterConfig.sharedInstance.playerConfig.enablePIPWhenExit;
        }
        return YES;
    };
    livePullVC.config.isCloseWhileDisappear = ^BOOL(BDLLivePullViewController * _Nonnull viewController) {
        return !viewController.isFlutterViewAlive;
    };
    livePullVC.config.autoCloseFloatingPlayerWhenAppear = !autoShowView;
}

- (void)leaveLiveRoomLogically:(NSDictionary *)arguments result:(FlutterResult)result {
    [self leaveLiveRoom:arguments result:result];
}

- (void)leaveLiveRoom:(NSDictionary *)arguments result:(FlutterResult)result {
    if (self.activity) {
        BDLLiveEngine *engine = [BDLLiveEngine sharedInstance];
        BDLLivePullViewController *livePullVC = [engine getLivePullViewController];
        
        if ([livePullVC isFloating]) {
            [livePullVC hideFloatingPlayerIfAvailable:NO];
        }
        else if (self.delegate.autoShowView) {
            [self.delegate hideLivePullViewController:livePullVC];
        }
        
        [engine leaveLiveRoom];
        self.activity = nil;
    }
    result(nil);
}

- (void)setLiveRoomStatusListener:(NSDictionary *)arguments result:(FlutterResult)result {
    if ([arguments[@"hasListener"] boolValue]) {
        self.delegate = [BdliveFlutterViewerDelegate eventHandlerWithName:@"com.bytedance.flutter_viewer.listener.event" binaryMessenger:[self.registrar messenger]];
    }
    else {
        [self.delegate destroy];
        self.delegate = nil;
    }
    result(nil);
}

- (void)hideLiveRoom:(NSDictionary *)arguments result:(FlutterResult)result {
    [[[BDLLiveEngine sharedInstance] getLivePullViewController] close:BdliveFlutterConfig.sharedInstance.playerConfig.enablePIP];
    result(nil);
}

- (void)openUrl:(NSDictionary *)arguments result:(FlutterResult)result {
    NSString *url = arguments[@"url"];
    BOOL enableFloating = [arguments[@"enableFloating"] boolValue];
    
    if (url) {
        [[[BDLLiveEngine sharedInstance] getLivePullViewController] defaultOpenURLAction:url enableFloating:enableFloating];
    }
    
    result(nil);
}

- (void)closeFloatingView:(NSDictionary *)arguments result:(FlutterResult)result {
    [[[BDLLiveEngine sharedInstance] getLivePullViewController] defaultFloatingPlayerCloseAction];
    result(nil);
}

// MARK: - player
- (void)createPlayer:(NSDictionary *)arguments result:(FlutterResult)result {
    int64_t insId = [arguments[@"managerId"] longLongValue];
    BdliveFlutterViewerPlayer *player = [[BdliveFlutterViewerPlayer alloc] initWithRegistrar:self.registrar insId:insId];
    if (!self.playerMap) {
        self.playerMap = [NSMutableDictionary dictionary];
    }
    BDLActivity *activity = [[BDLActivity alloc] init];
    
    activity.activityId = arguments[@"activityId"];
    activity.token = arguments[@"token"];
    activity.authMode = [arguments[@"authMode"] integerValue];
    activity.isPortrait = [arguments[@"isPortrait"] boolValue];
    player.activity = activity;
    
    self.playerMap[@(insId)] = player;
    __weak typeof(self) weakSelf = self;
    player.destroyBlock = ^(BdliveFlutterViewerPlayer * _Nonnull player) {
        typeof(self) self = weakSelf;
        [self.playerMap removeObjectForKey:@(insId)];
    };
    result(nil);
}

- (void)setLiveRoomConfig:(NSDictionary *)arguments result:(FlutterResult)result {
    BdliveFlutterConfig *config = [BdliveFlutterConfig yy_modelWithJSON:arguments];
    BdliveFlutterConfig.sharedInstance.playerConfig = config.playerConfig;
    [BdliveFlutterConfig.sharedInstance saveConfig];
}

@end
