//
//  BdliveFlutterStreamHandler.m
//
//  bdlive-flutter-viewer SDK License
//
//  Copyright 2025 Byteplus Pte. Ltd. All Rights Reserved.
//
//  The bdlive-flutter-viewer SDK was developed by Byteplus Pte. Ltd. (hereinafter “Byteplus”).
//  Any copyright or patent right is owned by and proprietary material of the Byteplus.
//
//  bdlive-flutter-viewer SDK is available under the Byteplus product and licensed under the commercial license.
//  Customers can contact via https://www.byteplus.com/en/contact for commercial licensing options.
//  Here is also a link to Service Level Agreement: https://docs.byteplus.com/en/byteplus-livesaas/docs/byteplus-live-service-level-agreement
//
//  Without Byteplus's prior written permission, any use of bdlive-flutter-viewer SDK, in particular any use for commercial purposes, is prohibited.
//  This includes, without limitation, incorporation in a commercial product, use in a commercial service, or production of other artefacts for commercial purposes.
//
//  Without Byteplus's prior written permission, the bdlive-flutter-viewer SDK may not be reproduced, modified and/or made available in any form to any third party.

#import "BdliveFlutterStreamHandler.h"
#import <objc/message.h>

static inline void dispatch_main_async_safe(dispatch_block_t block) {
    if (strcmp(dispatch_queue_get_label(DISPATCH_CURRENT_QUEUE_LABEL), dispatch_queue_get_label(dispatch_get_main_queue())) == 0) {
        block();
    } else {
        dispatch_async(dispatch_get_main_queue(), block);
    }
}

static NSString * const kMethodName = @"methodName";

@interface BdliveFlutterMethodHandler ()

@property (nonatomic, strong, nullable) FlutterMethodChannel *methodChannel;
@property (nonatomic, weak, nullable) id methodTarget;

@end

@implementation BdliveFlutterMethodHandler

+ (instancetype)methodChannelWithName:(NSString *)name methodTarget:(id)methodTarget binaryMessenger:(NSObject<FlutterBinaryMessenger> *)messenger {
    BdliveFlutterMethodHandler *handler = [[self alloc] init];
    handler.methodTarget = methodTarget;
    handler.methodChannel = [[FlutterMethodChannel alloc] initWithName:name
                                                       binaryMessenger:messenger
                                                                 codec:[FlutterStandardMethodCodec sharedInstance]];
    __weak typeof(handler) weakHandler = handler;
    [handler.methodChannel setMethodCallHandler:^(FlutterMethodCall *call, FlutterResult result) {
        dispatch_main_async_safe(^{
            [weakHandler handleMethodCall:call result:result];
        });
    }];
    
    return handler;
}

- (void)destroy {
    [self.methodChannel setMethodCallHandler:nil];
    self.methodChannel = nil;
    self.methodTarget = nil;
}

- (void)handleMethodCall:(FlutterMethodCall*)call result:(FlutterResult)result {
    SEL selector = NSSelectorFromString([call.method stringByAppendingString:@":result:"]);
    if ([self.methodTarget respondsToSelector:selector]) {
#if DEBUG
        NSLog(@"flutter call: %@ %@", NSStringFromSelector(selector), call.arguments);
#endif
        ((void(*)(id,SEL,id,FlutterResult))objc_msgSend)(self.methodTarget, selector, call.arguments, result);
        return;
    }
    result(FlutterMethodNotImplemented);
}

@end

@interface BdliveFlutterEventHandler ()

@property (nonatomic, strong, nullable) FlutterEventChannel *eventChannel;
@property (nonatomic, copy, nullable) FlutterEventSink eventSink;

@end

@implementation BdliveFlutterEventHandler

+ (instancetype)eventHandlerWithName:(NSString *)name binaryMessenger:(NSObject<FlutterBinaryMessenger> *)messenger {
    BdliveFlutterEventHandler *handler = [[self alloc] init];
    handler.eventChannel = [FlutterEventChannel eventChannelWithName:name
                                                     binaryMessenger:messenger];
    [handler.eventChannel setStreamHandler:handler];
    
    return handler;
}

- (void)emitEvent:(id _Nullable)event methodName:(nonnull NSString *)methodName {
    NSMutableDictionary *message = [NSMutableDictionary dictionary];
    message[kMethodName] = methodName;
    if ([event isKindOfClass:[NSDictionary class]]) {
        [message addEntriesFromDictionary:event];
    }
    
    dispatch_main_async_safe(^{
        if(self.eventSink) {
            self.eventSink(message);
        }
    });
}

- (void)destroy {
    [self.eventChannel setStreamHandler:nil];
    self.eventChannel = nil;
    self.eventSink = nil;
}

#pragma mark - FlutterStreamHandler
- (FlutterError * _Nullable)onCancelWithArguments:(id _Nullable)arguments {
    self.eventSink = nil;
    return nil;
}

- (FlutterError * _Nullable)onListenWithArguments:(id _Nullable)arguments eventSink:(nonnull FlutterEventSink)events {
    self.eventSink = events;
    return nil;
}

@end
