//
//  live_room_status_listener_example.dart
//
//  bdlive-flutter-viewer SDK License
//
//  Copyright 2025 Byteplus Pte. Ltd. All Rights Reserved.
//
//  The bdlive-flutter-viewer SDK was developed by Byteplus Pte. Ltd. (hereinafter “Byteplus”).
//  Any copyright or patent right is owned by and proprietary material of the Byteplus.
//
//  bdlive-flutter-viewer SDK is available under the Byteplus product and licensed under the commercial license.
//  Customers can contact via https://www.byteplus.com/en/contact for commercial licensing options.
//  Here is also a link to Service Level Agreement: https://docs.byteplus.com/en/byteplus-livesaas/docs/byteplus-live-service-level-agreement
//
//  Without Byteplus's prior written permission, any use of bdlive-flutter-viewer SDK, in particular any use for commercial purposes, is prohibited.
//  This includes, without limitation, incorporation in a commercial product, use in a commercial service, or production of other artefacts for commercial purposes.
//
//  Without Byteplus's prior written permission, the bdlive-flutter-viewer SDK may not be reproduced, modified and/or made available in any form to any third party.
import 'package:bdlive_flutter_viewer/bdlive_flutter_viewer.dart';
import 'package:flutter/material.dart';

import 'product_detail_page.dart';
import 'liveroom_config.dart';
import 'global.dart';
import 'live_room_page.dart';

class BdlLiveRoomStatusListenerExampleImpl
    implements BdlLiveRoomStatusListener {
  static final BdlLiveRoomStatusListenerExampleImpl _instance =
      BdlLiveRoomStatusListenerExampleImpl();
  static BdlLiveRoomStatusListenerExampleImpl get instance => _instance;

  bool autoShowView = false;
  MaterialPageRoute? _livePullRoute;
  BuildContext? _liveRouteContext;
  late BdlActivityConfig activity;

  // Initialize all callback properties as null first
  @override
  void Function(BdlActivityConfig activity)? onLiveRoomActivityResume;

  @override
  void Function(BdlActivityConfig activity)? onLiveRoomActivityPause;

  @override
  void Function(BdlActivityConfig activity)? onLiveRoomFullDestroyed;

  @override
  void Function(BdlActivityConfig activity)? onFloatViewWillAppear;

  @override
  void Function(BdlActivityConfig activity)? onFloatViewWillClose;

  @override
  void Function(String url, BdlViewerOpenUrlType type, bool enableFloating)?
      onOpenUrlCallback;

  @override
  void Function(BdlActivityConfig activity)? showLivePullView;

  @override
  void Function(BdlActivityConfig activity)? hideLivePullView;

  BdlLiveRoomStatusListenerExampleImpl() {
    onLiveRoomActivityResume = (activity) {
      debugPrint('onLiveRoomActivityResume...$activity');
      if (ProductDetailPage.pageIsExist && autoShowView) {
        navigatorKey.currentState!.popUntil((route) {
          return route.settings.name == 'SettingActivityPage';
        });
        ProductDetailPage.pageIsExist = false;
      }
    };

    onLiveRoomActivityPause = (activity) {
      debugPrint('onLiveRoomActivityPause...$activity');
    };

    onLiveRoomFullDestroyed = (activity) {
      debugPrint('onLiveRoomFullDestroyed...$activity');
      BdliveViewerApi.instance.setLiveRoomStatusListener(null);
    };

    onFloatViewWillAppear = (activity) {
      debugPrint('onFloatViewWillAppear...$activity');
    };

    onFloatViewWillClose = (activity) {
      debugPrint('onFloatViewWillClose...$activity');
    };

    onOpenUrlCallback = (url, type, enableFloating) {
      debugPrint('onOpenUrlCallback $url');
      debugPrint('onOpenUrlCallback ${LiveRoomConfig().openUrlType}');
      if (LiveRoomConfig().openUrlType == kOpenUrlTypeNewPage) {
        // Open new page
        showCardPage(activity, url, type, enableFloating);
      } else {
        // SDK default opening
        BdliveViewerApi.instance.openUrl(url, type, enableFloating);
      }
    };

    showLivePullView = (activity) {
      debugPrint('showLivePullView...$activity');
      if (autoShowView) {
        return;
      }
      // If LiveRoomPage already exists, remove pages above it and bring it to the top
      if (isLiveRoomRouteInStack()) {
        navigatorKey.currentState!.popUntil((route) {
          return route.settings.name == 'LiveRoomPage';
        });
        return;
      }
      // If the LiveRoomPage is not existed, create it
      _livePullRoute = MaterialPageRoute(
        builder: (context) {
          // Save context for later use
          _liveRouteContext = context;
          return const LiveRoomPage();
        },
        settings: const RouteSettings(name: 'LiveRoomPage'),
      );
      navigatorKey.currentState!.push(_livePullRoute!);
    };

    hideLivePullView = (activity) {
      debugPrint('hideLivePullView...$activity');
      if (autoShowView) {
        return;
      }
      if (_livePullRoute != null) {
        try {
          // Safely check and remove route
          if (isLiveRoomRouteAtTop()) {
            // If route is at top of stack, use pop
            navigatorKey.currentState!.pop();
          } else if (isLiveRoomRouteInStack()) {
            // If route is in stack but not at top, use removeRoute
            navigatorKey.currentState!.removeRoute(_livePullRoute!);
          }
          // Reset context reference
          _liveRouteContext = null;
          _livePullRoute = null;
        } catch (e) {
          debugPrint('Failed to remove LiveRoom route: $e');
          // Reset reference
          _liveRouteContext = null;
          _livePullRoute = null;
        }
      }
    };
  }

  /// Check if LiveRoom route is in the navigation stack
  bool isLiveRoomRouteInStack() {
    try {
      // Use ModalRoute.of to get the route associated with the context
      final modalRoute = ModalRoute.of(_liveRouteContext!);
      // If modalRoute is not null and isActive is true, the route is in the stack
      return modalRoute != null && modalRoute.isActive;
    } catch (e) {
      // Catch possible exceptions, such as context being invalid
      debugPrint('Error checking if LiveRoom route is in stack: $e');
      return false;
    }
  }

  /// Check if LiveRoom route is at the top of navigation stack
  bool isLiveRoomRouteAtTop() {
    try {
      // Use ModalRoute.of to get the route associated with the context
      final modalRoute = ModalRoute.of(_liveRouteContext!);
      // If modalRoute is not null and isCurrent is true, the route is at the top
      return modalRoute != null && modalRoute.isCurrent;
    } catch (e) {
      // Catch possible exceptions, such as context being invalid
      debugPrint('Error checking if LiveRoom route is at top: $e');
      return false;
    }
  }

  void showCardPage(BdlActivityConfig activity, String url,
      BdlViewerOpenUrlType type, bool enableFloating) {
    navigatorKey.currentState!.push(MaterialPageRoute(builder: (ctx) {
      return ProductDetailPage(
        activity: activity,
        url: url,
        type: type,
        enableFloating: enableFloating,
        isShowBackArrow: false,
        isAutoShowView: autoShowView,
      );
    }));
    ProductDetailPage.pageIsExist = true;

    if (autoShowView) {
      BdliveViewerApi.instance.hideLiveRoom();
    }
  }
}
