//
//  play_page.dart
//
//  bdlive-flutter-viewer SDK License
//
//  Copyright 2025 Byteplus Pte. Ltd. All Rights Reserved.
//
//  The bdlive-flutter-viewer SDK was developed by Byteplus Pte. Ltd. (hereinafter “Byteplus”).
//  Any copyright or patent right is owned by and proprietary material of the Byteplus.
//
//  bdlive-flutter-viewer SDK is available under the Byteplus product and licensed under the commercial license.
//  Customers can contact via https://www.byteplus.com/en/contact for commercial licensing options.
//  Here is also a link to Service Level Agreement: https://docs.byteplus.com/en/byteplus-livesaas/docs/byteplus-live-service-level-agreement
//
//  Without Byteplus's prior written permission, any use of bdlive-flutter-viewer SDK, in particular any use for commercial purposes, is prohibited.
//  This includes, without limitation, incorporation in a commercial product, use in a commercial service, or production of other artefacts for commercial purposes.
//
//  Without Byteplus's prior written permission, the bdlive-flutter-viewer SDK may not be reproduced, modified and/or made available in any form to any third party.
import 'dart:ffi';

import 'package:bdlive_flutter_viewer/bdlive_flutter_viewer.dart';
import 'package:flutter/material.dart';
import '../common_radio_form_item.dart';
import '../l10n/app_localizations.dart';

class PlayerPage extends StatefulWidget {
  final BdlViewerPlayerConfig activity;

  PlayerPage({super.key, required this.activity});

  @override
  State<PlayerPage> createState() => _PlayerPageState();
}

class _PlayerPageState extends State<PlayerPage> with WidgetsBindingObserver {
  BdliveViewerPlayer? _player;
  final List<String> _items = [];
  bool _isMute = false;
  bool _isLoop = false;
  BdlivePlayerLayoutMode _fillMode = BdlivePlayerLayoutMode.aspectFit;
  double _value = 1.0;
  double _totalDuration = 0.0;
  double _seekDuration = 0.0;
  double _currentPlaybackTime = 0.0;
  bool _hideCover = false;
  List<BdliveVideoResolution> _supportedResolutions = [];

  @override
  void initState() {
    super.initState();
    _initPlayer();
    WidgetsBinding.instance.addObserver(this);
  }

  @override
  void didChangeAppLifecycleState(AppLifecycleState state) {
    super.didChangeAppLifecycleState(state);
    if (state == AppLifecycleState.paused) {
      print('应用进入后台');
      _player?.setInBackground(true);
    } else if (state == AppLifecycleState.resumed) {
      print('应用回到前台');
      _player?.setInBackground(false);
    }
  }

  void _addItem(String text) {
    print(text);
    if (mounted) {
      setState(() {
        _items.add(text);
      });
    }
  }

  void _updateValue(double newValue) {
    String str = newValue.toStringAsFixed(2);
    double speed = double.parse(str);
    _player?.setVodPlaySpeed(speed);
    setState(() {
      _value = speed;
    });
  }

  void _seekChanged(double duration) async {
    String str = duration.toStringAsFixed(2);
    double value = double.parse(str);
    setState(() {
      _seekDuration = value;
    });
  }

  void _seekChangeEnd(double duration) async {
    double value = _fixedTwoNum(duration);
    int num = (value * 1000).toInt();
    bool result = await _player?.seekVodTime(num) ?? false;
    _addItem('seek result: $result');
  }

  double _fixedTwoNum(double value) {
    String str = value.toStringAsFixed(2);
    return double.parse(str);
  }

  void _initPlayer() async {
    // 创建 player 实例
    BdliveViewerPlayer player =
        await BdliveViewerPlayer.createPlayer(widget.activity);

    BdliveViewerPlayerDelegate delegate = BdliveViewerPlayerDelegate(
      liveRoomStatusChanged: (status) {
        _addItem('liveRoomStatusChanged: $status');
      },
      playableStatusChanged: (playableStatus) {
        _addItem('playableStatusChanged: $playableStatus');
      },
      isPlayingChanged: (isPlaying) {
        _addItem('isPlayingChanged: $isPlaying');
      },
      videoSizeChanged: (width, height) {
        _addItem('videoSizeChanged: $width, $height');
      },
      isStallingChanged: (isStalling) {
        _addItem('isStallingChanged: $isStalling');
      },
      vodErrorOccurred: (error) {
        _addItem('vodErrorOccurred: $error');
      },
      liveErrorOccurred: (error) {
        _addItem('liveErrorOccurred: $error');
      },
      playErrorStatusChanged: (isPlayError) {
        _addItem('playErrorStatusChanged: $isPlayError');
      },
      vodPrepared: () {
        _addItem('vodPrepared');
        _caculateDuration();
      },
      vodRenderStarted: () {
        _addItem('vodRenderStarted');
      },
      vodAutoSeekPreviousTime: (seekTimeInMills) {
        _addItem('vodAutoSeekPreviousTime: $seekTimeInMills');
      },
      vodCurPlayTimeChanged: (curTimeInMills) {
        // _addItem('vodCurPlayTimeChanged: $curTimeInMills');
        final curTime = curTimeInMills / 1000;
        final time = _fixedTwoNum(curTime);
        setState(() {
          _currentPlaybackTime = time;
          _seekDuration = time;
        });
      },
      vodDurationChanged: (durationInMills) {
        _addItem('vodDurationChanged: $durationInMills');
      },
      livePrepared: () {
        _addItem('livePrepared');
      },
      liveFirstFrameRendered: (isFirstFrame) {
        _addItem('liveFirstFrameRendered: $isFirstFrame');
      },
      coverImageVisibleChanged: (isVisible) {
        _addItem('coverImageVisibleChanged: $isVisible');
      },
      resolutionInfoChanged: (resolutions, defaultResolution) {
        setState(() {
          _supportedResolutions = resolutions;
        });
        _addItem('resolutionInfoChanged: $resolutions, $defaultResolution');
      },
      coverImageSizeChanged: (width, height) {
        _addItem('coverImageSizeChanged: $width, $height');
      },
      onUserForceOffline: (reason) {
        _addItem('onUserForceOffline: $reason');
      },
      onRelease: () {
        _addItem('onRelease');
      },
    );
    player.setDelegate(delegate);

    setState(() {
      _player = player;
    });
  }

  void _caculateDuration() async {
    int duration = await _player?.getCurVodDuration() ?? 0;
    setState(() {
      _totalDuration = _fixedTwoNum(duration / 1000);
    });
  }

  String _formatDuration(double seconds) {
    int totalSeconds = seconds.toInt();
    return totalSeconds.toString();
  }

  @override
  Widget build(BuildContext context) {
    return Scaffold(
      appBar: AppBar(
        title: Text(AppLocalizations.of(context)!.purePlayerView),
        actions: _supportedResolutions.length == 0
            ? null
            : [
                TextButton(
                    onPressed: () async {
                      final curResolution = await _player?.getCurResolution();
                      if (curResolution == null) {
                        return;
                      }

                      showModalBottomSheet(
                        context: context,
                        builder: (BuildContext context) {
                          return Container(
                            child: Column(
                              children: _supportedResolutions
                                  .map((e) => ListTile(
                                        title: Text(
                                          e.toString(),
                                          style: TextStyle(
                                              color: curResolution == e
                                                  ? Colors.red
                                                  : Colors.black),
                                        ),
                                        onTap: () {
                                          Navigator.pop(context);
                                          _addItem('setCurResolution: $e');
                                          _player?.setCurResolution(e);
                                        },
                                      ))
                                  .toList(),
                            ),
                          );
                        },
                      );
                    },
                    child: Text(AppLocalizations.of(context)!.resolution)),
              ],
      ),
      body: Column(
        children: [
          SizedBox(height: 10),
          Wrap(
            spacing: 5,
            runSpacing: 5,
            children: [
              _buildButton('play', () {
                _player?.play();
              }),
              _buildButton('pause', () {
                _player?.pause();
              }),
              _buildButton(_isMute ? 'unMute' : 'mute', () async {
                if (_isMute) {
                  await _player?.setMute(false);
                } else {
                  await _player?.setMute(true);
                }
                setState(() {
                  _isMute = !_isMute;
                });
              }),
              _buildButton(_isLoop ? 'unLoop' : 'loop', () async {
                if (_isLoop) {
                  await _player?.setVodLoop(false);
                } else {
                  await _player?.setVodLoop(true);
                }
                setState(() {
                  _isLoop = !_isLoop;
                });
              }),
              _buildButton(AppLocalizations.of(context)!.refreshLiveContent, () async {
                if (_player == null) {
                  return;
                }
                BdlivePlayableStatus status =
                    await _player!.getPlayableStatus();
                if (status == BdlivePlayableStatus.liveCanPlay) {
                  _player?.refreshLive();
                }
              }),
              _buildButton(_hideCover ? AppLocalizations.of(context)!.showCover : AppLocalizations.of(context)!.hideCover, () async {
                await _player?.setCoverImageContainerVisibility(_hideCover);
                setState(() {
                  _hideCover = !_hideCover;
                });
              }),
            ],
          ),
          CommonRadioFormItem(
            AppLocalizations.of(context)!.fillMode,
            options: const ['a-fit', 'a-fill', 'fill'],
            values: const [
              BdlivePlayerLayoutMode.aspectFit,
              BdlivePlayerLayoutMode.aspectFill,
              BdlivePlayerLayoutMode.fill,
            ],
            groupValue: _fillMode,
            onChanged: (val) {
              setState(() {
                _fillMode = val as BdlivePlayerLayoutMode;
              });
              _player?.setPlayerLayoutMode(_fillMode);
            },
          ),
          Column(
            crossAxisAlignment: CrossAxisAlignment.start,
            children: <Widget>[
              Padding(
                padding: const EdgeInsets.only(left: 10),
                child: Text('${AppLocalizations.of(context)!.playSpeed}: $_value'),
              ),
              Slider(
                value: _value,
                min: 0.1,
                max: 2.0,
                onChanged: _updateValue,
              ),
            ],
          ),
          if (_totalDuration > 0)
            Column(
              crossAxisAlignment: CrossAxisAlignment.start,
              children: <Widget>[
                Padding(
                  padding: const EdgeInsets.only(left: 10),
                  child: Text(
                      '${AppLocalizations.of(context)!.currentPlaybackTime}: ${_formatDuration(_currentPlaybackTime)}s / 视频总时长:${_formatDuration(_totalDuration)}s'),
                ),
                Row(
                  children: [
                    Expanded(
                      child: Slider(
                        value: _seekDuration,
                        min: 0.0,
                        max: _totalDuration,
                        onChanged: _seekChanged,
                        onChangeEnd: _seekChangeEnd,
                      ),
                    ),
                    Text('${AppLocalizations.of(context)!.seekTo}: ${_formatDuration(_seekDuration)}s'),
                    SizedBox(
                      width: 10,
                    ),
                  ],
                ),
              ],
            ),
          Container(
              width: MediaQuery.of(context).size.width,
              height: MediaQuery.of(context).size.width * (6 / 16),
              color: _player != null ? Colors.green : Colors.yellow,
              child: _player != null
                  ? BdliveViewerView(player: _player!, onPlatformViewCreated: (){
                    _player?.start();
              },)
                  : SizedBox()),
          const SizedBox(
            height: 2,
          ),
          Expanded(
            child: Container(
              color: Colors.cyan,
              child: ListView.builder(
                itemCount: _items.length,
                itemBuilder: (context, index) {
                  return Text(_items[index]);
                },
              ),
            ),
          ),
        ],
      ),
    );
  }

  Widget _buildButton(String text, void Function() onPressed) {
    return InkWell(
      onTap: onPressed,
      child: Container(
        padding: EdgeInsets.symmetric(horizontal: 10, vertical: 5),
        decoration: BoxDecoration(
          color: Colors.grey,
          borderRadius: BorderRadius.circular(20),
        ),
        child: Text(
          text,
          style: const TextStyle(fontSize: 16, color: Colors.white),
        ),
      ),
    );
  }

  @override
  void dispose() {
    WidgetsBinding.instance.removeObserver(this);
    super.dispose();
    _player?.destroy();
  }
}
