//
//  feed_page.dart
//
//  bdlive-flutter-viewer SDK License
//
//  Copyright 2025 Byteplus Pte. Ltd. All Rights Reserved.
//
//  The bdlive-flutter-viewer SDK was developed by Byteplus Pte. Ltd. (hereinafter “Byteplus”).
//  Any copyright or patent right is owned by and proprietary material of the Byteplus.
//
//  bdlive-flutter-viewer SDK is available under the Byteplus product and licensed under the commercial license.
//  Customers can contact via https://www.byteplus.com/en/contact for commercial licensing options.
//  Here is also a link to Service Level Agreement: https://docs.byteplus.com/en/byteplus-livesaas/docs/byteplus-live-service-level-agreement
//
//  Without Byteplus's prior written permission, any use of bdlive-flutter-viewer SDK, in particular any use for commercial purposes, is prohibited.
//  This includes, without limitation, incorporation in a commercial product, use in a commercial service, or production of other artefacts for commercial purposes.
//
//  Without Byteplus's prior written permission, the bdlive-flutter-viewer SDK may not be reproduced, modified and/or made available in any form to any third party.
import 'package:bdlive_flutter_viewer/bdlive_flutter_viewer.dart';
import 'package:flutter/material.dart';
import '../common_radio_form_item.dart';
import '../product_detail_page.dart';
import '../l10n/app_localizations.dart';

class FeedPage extends StatefulWidget {
  final BdlViewerPlayerConfig activity;

  FeedPage({super.key, required this.activity});

  @override
  State<FeedPage> createState() => _FeedPageState();
}

class _FeedPageState extends State<FeedPage> {
  BdliveViewerPlayer? _player;

  double _currentPlaybackTime = 0.0;
  double _totalDuration = 0.0;
  double _seekDuration = 0.0;

  bool _isPlaying = true;
  double _currentSpeed = 1.0;
  List<double> _speeds = [0.5, 1.0, 1.5, 2.0];
  List<BdliveVideoResolution> _supportedResolutions = [];

  BdLiveRoomStatus _playableStatus = BdLiveRoomStatus.unknown;

  @override
  void initState() {
    super.initState();
    _initPlayer();
  }

  void _seekChanged(double duration) async {
    String str = duration.toStringAsFixed(2);
    double value = double.parse(str);
    setState(() {
      _seekDuration = value;
    });
  }

  void _seekChangeEnd(double duration) async {
    double value = _fixedTwoNum(duration);
    int num = (value * 1000).toInt();
    await _player?.seekVodTime(num);
  }

  double _fixedTwoNum(double value) {
    String str = value.toStringAsFixed(2);
    return double.parse(str);
  }

  void _addItem(String text) {
    print(text);
  }

  void _initPlayer() async {
    // 创建 player 实例
    BdliveViewerPlayer player =
        await BdliveViewerPlayer.createPlayer(widget.activity);

    BdliveViewerPlayerDelegate delegate = BdliveViewerPlayerDelegate(
      liveRoomStatusChanged: (status) {
        setState(() {
          _playableStatus = status;
        });
        _addItem('liveRoomStatusChanged: $status');
      },
      playableStatusChanged: (playableStatus) {
        _addItem('playableStatusChanged: $playableStatus');
      },
      isPlayingChanged: (isPlaying) {
        setState(() {
          _isPlaying = isPlaying;
        });
        _addItem('isPlayingChanged: $isPlaying');
      },
      videoSizeChanged: (width, height) {
        _addItem('videoSizeChanged: $width, $height');
      },
      isStallingChanged: (isStalling) {
        _addItem('isStallingChanged: $isStalling');
      },
      vodErrorOccurred: (error) {
        _addItem('vodErrorOccurred: $error');
      },
      liveErrorOccurred: (error) {
        _addItem('liveErrorOccurred: $error');
      },
      playErrorStatusChanged: (isPlayError) {
        _addItem('playErrorStatusChanged: $isPlayError');
      },
      vodPrepared: () {
        _addItem('vodPrepared');
        _caculateDuration();
      },
      vodRenderStarted: () {
        _addItem('vodRenderStarted');
      },
      vodAutoSeekPreviousTime: (seekTimeInMills) {
        _addItem('vodAutoSeekPreviousTime: $seekTimeInMills');
      },
      vodCurPlayTimeChanged: (curTimeInMills) {
        // _addItem('vodCurPlayTimeChanged: $curTimeInMills');
        final curTime = curTimeInMills / 1000;
        final time = _fixedTwoNum(curTime);
        setState(() {
          _currentPlaybackTime = time;
          _seekDuration = time;
        });
      },
      vodDurationChanged: (durationInMills) {
        _addItem('vodDurationChanged: $durationInMills');
      },
      livePrepared: () {
        _addItem('livePrepared');
      },
      liveFirstFrameRendered: (isFirstFrame) {
        _addItem('liveFirstFrameRendered: $isFirstFrame');
      },
      coverImageVisibleChanged: (isVisible) {
        _addItem('coverImageVisibleChanged: $isVisible');
      },
      resolutionInfoChanged: (resolutions, defaultResolution) {
        setState(() {
          _supportedResolutions = resolutions;
        });
        _addItem('resolutionInfoChanged: $resolutions, $defaultResolution');
      },
      coverImageSizeChanged: (width, height) {
        _addItem('coverImageSizeChanged: $width, $height');
      },
      onUserForceOffline: (reason) {
        _addItem('onUserForceOffline: $reason');
      },
      onRelease: () {
        _addItem('onRelease');
      },
    );
    player.setDelegate(delegate);

    _player = player;
    setState(() {
      _player = player;
    });
  }

  void _caculateDuration() async {
    int duration = await _player?.getCurVodDuration() ?? 0;
    setState(() {
      _totalDuration = _fixedTwoNum(duration / 1000);
    });
  }

  String _formatDuration(double seconds) {
    int totalSeconds = seconds.toInt();
    // 创建 Duration 对象
    Duration duration = Duration(seconds: totalSeconds);

    // 计算小时、分钟和剩余的秒数
    String twoDigits(int n) => n.toString().padLeft(2, "0");
    String twoDigitMinutes = twoDigits(duration.inMinutes.remainder(60));
    String twoDigitSeconds = twoDigits(duration.inSeconds.remainder(60));
    int hours = duration.inHours;

    // 根据是否有小时来构建返回字符串
    if (hours > 0) {
      return "${hours}:${twoDigitMinutes}:${twoDigitSeconds}";
    } else {
      return "${twoDigitMinutes}:${twoDigitSeconds}";
    }
  }

  String getPlayableStatus(BdLiveRoomStatus status) {
    switch (status) {
      case BdLiveRoomStatus.unknown:
        return AppLocalizations.of(context)!.unknown;
      case BdLiveRoomStatus.live:
        return AppLocalizations.of(context)!.live;
      case BdLiveRoomStatus.preview:
        return AppLocalizations.of(context)!.preview;
      case BdLiveRoomStatus.replay:
        return AppLocalizations.of(context)!.replay;
      default:
        return AppLocalizations.of(context)!.unknown;
    }
  }

  @override
  Widget build(BuildContext context) {
    return Scaffold(
      body: GestureDetector(
        onTap: () async {
          print("GestureDetector onTap");
          bool isPlaying = await _player?.isPlaying() ?? false;
          isPlaying ? _player?.pause() : _player?.play();
        },
        child: Container(
          width: MediaQuery.of(context).size.width,
          height: MediaQuery.of(context).size.height,
          decoration: const BoxDecoration(color: Colors.black),
          child: Stack(
            children: [
              Container(
                  child: _player != null
                      ? BdliveViewerView(
                          player: _player!,
                          onPlatformViewCreated: () {
                            _player?.start();
                          },
                        )
                      : SizedBox()),
              Container(
                width: MediaQuery.of(context).size.width,
                height: MediaQuery.of(context).size.height,
                decoration: const BoxDecoration(color: Colors.transparent),
              ),
              Center(child: isPlaying()),
              Positioned(
                  left: 0,
                  right: 0,
                  top: 100,
                  child: Row(
                    mainAxisAlignment: MainAxisAlignment.center,
                    children: [
                      GestureDetector(
                        onTap: () {
                          if (_playableStatus == BdLiveRoomStatus.live) {
                            _player?.pause();
                            enterLiveRoom();
                          }
                        },
                        child: Container(
                          padding: const EdgeInsets.symmetric(
                              horizontal: 10, vertical: 5),
                          color: Colors.red,
                          child: Text(
                            'status: ${getPlayableStatus(_playableStatus)}',
                            style: TextStyle(color: Colors.white),
                          ),
                        ),
                      ),
                    ],
                  )),
              SafeArea(
                child: Column(
                  children: [
                    Spacer(),
                    Row(
                      children: [
                        if (_playableStatus != BdLiveRoomStatus.live)
                          Container(
                            // color: Colors.red,
                              width: 50,
                              child: Center(
                                child: Text(
                                                            _formatDuration(_currentPlaybackTime),
                                                            style: TextStyle(color: Colors.white),
                                                          ),
                              )),
                        if (_playableStatus != BdLiveRoomStatus.live)
                          Expanded(
                            child: Slider(
                              min: 0.0,
                              max: _totalDuration,
                              value: _seekDuration,
                              onChanged: _seekChanged,
                              onChangeEnd: _seekChangeEnd,
                              activeColor: Colors.green,
                              // 这个属性将影响滑块和激活轨道的颜色
                              inactiveColor: Colors.grey.shade400, // 非激活轨道的颜色
                            ),
                          ),
                        if (_playableStatus != BdLiveRoomStatus.live)
                          Text(
                            _formatDuration(_totalDuration),
                            style: TextStyle(color: Colors.white),
                          ),
                        if (_playableStatus != BdLiveRoomStatus.live)
                          TextButton(
                            style: TextButton.styleFrom(
                              padding: EdgeInsets.zero,
                              minimumSize: Size.zero,
                            ),
                            onPressed: () async {
                              showModalBottomSheet(
                                context: context,
                                builder: (BuildContext context) {
                                  return Container(
                                    child: Column(
                                      children: _speeds
                                          .map((e) => ListTile(
                                                title: Text(
                                                  e.toString(),
                                                  style: TextStyle(
                                                      color: _currentSpeed == e
                                                          ? Colors.red
                                                          : Colors.black),
                                                ),
                                                onTap: () {
                                                  Navigator.pop(context);
                                                  _player?.setVodPlaySpeed(e);
                                                  setState(() {
                                                    _currentSpeed = e;
                                                  });
                                                },
                                              ))
                                          .toList(),
                                    ),
                                  );
                                },
                              );
                            },
                            child: Text(
                              AppLocalizations.of(context)!.speed,
                              style: TextStyle(color: Colors.white),
                            ),
                          ),
                        if (_playableStatus == BdLiveRoomStatus.live) Spacer(),
                        if (_playableStatus == BdLiveRoomStatus.live)
                          TextButton(
                            style: TextButton.styleFrom(
                              padding: EdgeInsets.zero,
                              minimumSize: Size.zero,
                            ),
                            onPressed: () {
                              _player?.refreshLive();
                            },
                            child: Icon(Icons.refresh, color: Colors.white),
                          ),
                        TextButton(
                            style: TextButton.styleFrom(
                              padding: EdgeInsets.zero,
                              minimumSize: Size.zero,
                            ),
                            onPressed: () async {
                              final curResolution =
                                  await _player?.getCurResolution();
                              if (curResolution == null) {
                                return;
                              }
                              final resolutions =
                                  await _player?.getResolutions();
                              if (resolutions == null) {
                                return;
                              }
                              showModalBottomSheet(
                                context: context,
                                builder: (BuildContext context) {
                                  return Container(
                                    child: Column(
                                      children: resolutions
                                          .map((e) => ListTile(
                                                title: Text(
                                                  e.toString(),
                                                  style: TextStyle(
                                                      color: curResolution == e
                                                          ? Colors.red
                                                          : Colors.black),
                                                ),
                                                onTap: () {
                                                  Navigator.pop(context);
                                                  _player?.setCurResolution(e);
                                                },
                                              ))
                                          .toList(),
                                    ),
                                  );
                                },
                              );
                            },
                            child: Text(
                              AppLocalizations.of(context)!.resolution,
                              style: TextStyle(color: Colors.white),
                            )),
                      ],
                    ),
                  ],
                ),
              ),
            ],
          ),
        ),
      ),
    );
  }

  @override
  void dispose() {
    super.dispose();
    _player?.destroy();
  }

  Widget isPlaying() {
    return _isPlaying
        ? const SizedBox()
        : Image.asset(
            'assets/images/icon_play.png',
            width: 80,
            height: 80,
          );
  }

  void showCardPage(BdlActivityConfig activity, String url,
      BdlViewerOpenUrlType type, bool enableFloating) {
    Navigator.of(context).push(MaterialPageRoute(builder: (ctx) {
      return ProductDetailPage(
        activity: activity,
        url: url,
        type: type,
        enableFloating: enableFloating,
        isShowBackArrow: false,
        isAutoShowView: false,
      );
    }));
    ProductDetailPage.pageIsExist = true;

    BdliveViewerApi.instance.hideLiveRoom();
  }

  var _isLoading = false;

  void enterLiveRoom() async {
    setState(() {
      _isLoading = true;
    });

    BdlActivityConfig activity = BdlActivityConfig.fromMap(widget.activity.toMap());

    var isPreLoadSuccess =
        await BdliveViewerApi.instance.enterLiveRoom(activity);
    setState(() {
      _isLoading = false;
    });
    if (isPreLoadSuccess) {
      BdlLiveRoomStatusListener listener = BdlLiveRoomStatusListener(
        onLiveRoomActivityResume: (activity) {
          debugPrint('onLiveRoomActivityResume...$activity');
          if (ProductDetailPage.pageIsExist) {
            Navigator.of(context).pop();
            ProductDetailPage.pageIsExist = false;
          }
        },
        onLiveRoomActivityPause: (activity) {
          debugPrint('onLiveRoomActivityPause...$activity');
        },
        onLiveRoomFullDestroyed: (activity) {
          debugPrint('onLiveRoomFullDestroyed...$activity');
          BdliveViewerApi.instance.setLiveRoomStatusListener(null);
        },
        onFloatViewWillAppear: (activity) {
          debugPrint('onFloatViewWillAppear...$activity');
          // BdliveViewerApi.instance.leaveLiveRoom();
        },
        onFloatViewWillClose: (activity) {
          debugPrint('onFloatViewWillClose...$activity');
        },
        onOpenUrlCallback: (url, type, enableFloating) {
          debugPrint('onOpenUrlCallback $url');
          // Open new page
          showCardPage(activity, url, type, enableFloating);
        },
      );
      BdliveViewerApi.instance.setLiveRoomStatusListener(listener);
    }
  }
}
