//
//  SingleLiveConnector.java
//
//  bdlive-flutter-viewer SDK License
//
//  Copyright 2025 Byteplus Pte. Ltd. All Rights Reserved.
//
//  The bdlive-flutter-viewer SDK was developed by Byteplus Pte. Ltd. (hereinafter “Byteplus”).
//  Any copyright or patent right is owned by and proprietary material of the Byteplus.
//
//  bdlive-flutter-viewer SDK is available under the Byteplus product and licensed under the commercial license.
//  Customers can contact via https://www.byteplus.com/en/contact for commercial licensing options.
//  Here is also a link to Service Level Agreement: https://docs.byteplus.com/en/byteplus-livesaas/docs/byteplus-live-service-level-agreement
//
//  Without Byteplus's prior written permission, any use of bdlive-flutter-viewer SDK, in particular any use for commercial purposes, is prohibited.
//  This includes, without limitation, incorporation in a commercial product, use in a commercial service, or production of other artefacts for commercial purposes.
//
//  Without Byteplus's prior written permission, the bdlive-flutter-viewer SDK may not be reproduced, modified and/or made available in any form to any third party.
package com.example.bdlive_flutter_viewer.player;

import android.content.Context;
import android.view.View;

import androidx.annotation.NonNull;

import com.bytedance.live.common.utils.StringUtils;
import com.bytedance.live.sdk.player.TVULiveRoom;
import com.bytedance.live.sdk.player.TVULiveRoomServer;
import com.bytedance.live.sdk.player.listener.SinglePlayerListener;
import com.bytedance.live.sdk.player.model.vo.BDLForceOfflineInfo;
import com.bytedance.live.sdk.player.model.vo.BDLForceOfflineReason;
import com.bytedance.live.sdk.player.view.tvuSinglePlay.InitConfig;
import com.bytedance.live.sdk.player.view.tvuSinglePlay.TVUSinglePlayerView;
import com.example.bdlive_flutter_viewer.BdliveFlutterViewerPlugin;
import com.example.bdlive_flutter_viewer.FlutterEventChannelHelper;
import com.example.bdlive_flutter_viewer.vo.BDLVideoResolution;
import com.example.bdlive_flutter_viewer.vo.BdlActivity;
import com.example.bdlive_flutter_viewer.vo.FlutterError;
import com.ss.ttvideoengine.SeekCompletionListener;
import com.ss.ttvideoengine.TTVideoEngineInterface;
import com.ss.ttvideoengine.utils.Error;
import com.ss.videoarch.liveplayer.VeLivePlayerError;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import io.flutter.embedding.engine.plugins.FlutterPlugin;
import io.flutter.plugin.common.MethodCall;
import io.flutter.plugin.common.MethodChannel;

public class SingleLiveConnector {
    public static final int FlutterScaleAspectFit = 0;
    public static final int FlutterScaleAspectFill = 1;
    public static final int FlutterScaleFill = 2;

    static final HashMap<Integer, Integer> scaleModeMap = new HashMap<Integer, Integer>() {
        {
            put(FlutterScaleAspectFit, TTVideoEngineInterface.IMAGE_LAYOUT_ASPECT_FIT);
            put(FlutterScaleAspectFill, TTVideoEngineInterface.IMAGE_LAYOUT_ASPECT_FILL);
            put(FlutterScaleFill, TTVideoEngineInterface.IMAGE_LAYOUT_TO_FILL);
        }
    };

    private final Integer managerId;
    private final BdlActivity bdlActivity;
    private final FlutterLivePlayerManager manager;
    private final TVULiveRoomServer roomServer;
    private final TVUSinglePlayerView tvuSinglePlayerView;
    private MethodChannel methodChannel;
    private FlutterEventChannelHelper eventChannelHelper;
    private final FlutterPlugin.FlutterPluginBinding binding;
    private int bindPlayerViewId;

    private boolean hasSetDelegate;
    private HashMap<BDLVideoResolution, String> resolutionStringHashMap = new HashMap<>();

    private SinglePlayerListener singlePlayerListener = new SinglePlayerListener() {
        @Override
        public void liveRoomStatusChanged(int i) {
            Map<String, Object> map = new HashMap<>();
            map.put("status", i);
            sendEventToFlutter("liveRoomStatusChanged", map);
        }

        @Override
        public void playableStatusChanged(int i) {
            Map<String, Object> map = new HashMap<>();
            map.put("playableStatus", i);
            sendEventToFlutter("playableStatusChanged", map);
        }

        @Override
        public void playStatusChanged(int i) {
            Map<String, Object> map = new HashMap<>();
            map.put("isPlaying", i == 1);
            sendEventToFlutter("isPlayingChanged", map);
        }

        @Override
        public void sizeChanged(int i, int i1) {
            Map<String, Object> map = new HashMap<>();
            map.put("width", i);
            map.put("height", i1);
            sendEventToFlutter("videoSizeChanged", map);
        }

        @Override
        public void stallingStatusChanged(boolean b) {
            Map<String, Object> map = new HashMap<>();
            map.put("isStalling", b);
            sendEventToFlutter("isStallingChanged", map);
        }

        @Override
        public void vodErrorOccurred(Error error) {
            Map<String, Object> map = new HashMap<>();
            map.put("error", new FlutterError(error.code, error.toString()).generateFlutterMap());
            sendEventToFlutter("vodErrorOccurred", map);
        }

        @Override
        public void liveErrorOccurred(VeLivePlayerError veLivePlayerError) {
            Map<String, Object> map = new HashMap<>();
            map.put("error", new FlutterError(veLivePlayerError.mErrorCode, veLivePlayerError.mErrorMsg).generateFlutterMap());
            sendEventToFlutter("liveErrorOccurred", map);
        }

        @Override
        public void playErrorStatusChanged(boolean b) {
            Map<String, Object> map = new HashMap<>();
            map.put("isPlayError", b);
            sendEventToFlutter("playErrorStatusChanged", map);
        }

        @Override
        public void vodPrepared() {
            Map<String, Object> map = new HashMap<>();
            sendEventToFlutter("vodPrepared", map);
        }

        @Override
        public void vodRenderStarted() {
            Map<String, Object> map = new HashMap<>();
            sendEventToFlutter("vodRenderStarted", map);
        }

        @Override
        public void vodAutoSeekPreviousTime(int i) {
            Map<String, Object> map = new HashMap<>();
            map.put("seekTimeInMills", i);
            sendEventToFlutter("vodAutoSeekPreviousTime", map);
        }

        @Override
        public void vodCurPlayTimeChanged(int i) {
            Map<String, Object> map = new HashMap<>();
            map.put("curTimeInMills", i);
            sendEventToFlutter("vodCurPlayTimeChanged", map);
        }

        @Override
        public void vodDurationChanged(int i) {
            Map<String, Object> map = new HashMap<>();
            map.put("durationInMills", i);
            sendEventToFlutter("vodDurationChanged", map);
        }

        @Override
        public void vodCompletion() {

        }

        @Override
        public void livePrepared() {
            Map<String, Object> map = new HashMap<>();
            sendEventToFlutter("livePrepared", map);
        }

        @Override
        public void liveFirstFrameRendered(boolean b) {
            Map<String, Object> map = new HashMap<>();
            map.put("isFirstFrame", b);
            sendEventToFlutter("liveFirstFrameRendered", map);
        }

        @Override
        public void liveCompletion() {

        }

        @Override
        public void coverImageVisibleChanged(boolean b) {
            Map<String, Object> map = new HashMap<>();
            map.put("isVisible", b);
            sendEventToFlutter("coverImageVisibleChanged", map);
        }

        @Override
        public void resolutionInfoChanged(String[] strings, String s) {
            if (strings != null) {
                resolutionStringHashMap.clear();
                for (String string : strings) {
                    BDLVideoResolution videoResolution = BDLVideoResolution.castStr2BDLVideoResolution(string);
                    if (videoResolution != null) {
                        resolutionStringHashMap.put(videoResolution, string);
                    }
                }
                BDLVideoResolution defaultResolution = BDLVideoResolution.castStr2BDLVideoResolution(s);
                if (defaultResolution != null) {
                    Map<String, Object> map = new HashMap<>();
                    map.put("resolutions", BDLVideoResolution.castStrList2ResolutionIntList(strings));
                    map.put("defaultResolution", defaultResolution.value);
                    sendEventToFlutter("resolutionInfoChanged", map);
                }
            }
        }

        @Override
        public void onUserForceOffline(BDLForceOfflineInfo info) {
            Map<String, Object> map = new HashMap<>();
            map.put("reason", castBDLForceOfflineReason2Int(info.getReason()));
            sendEventToFlutter("onUserForceOffline", map);
        }

        @Override
        public void onRelease() {
            Map<String, Object> map = new HashMap<>();
            sendEventToFlutter("onRelease", map);
        }
    };

    public SingleLiveConnector(Integer managerId, BdlActivity bdlActivity, FlutterPlugin.FlutterPluginBinding binding,
                               FlutterLivePlayerManager manager) {
        this.managerId = managerId;
        this.bdlActivity = bdlActivity;
        this.manager = manager;
        this.binding = binding;

        //init tvuSinglePlayerView
        Context applicationContext = binding.getApplicationContext();
        roomServer = new TVULiveRoomServer(applicationContext, bdlActivity.getActivityId(), bdlActivity.getToken(), TVULiveRoomServer.SERVER_TYPE_SINGLE);
        roomServer.setRoomAuthMode(bdlActivity.getAuthMode() == 1 ? TVULiveRoom.TVURoomAuthMode.PUBLIC : TVULiveRoom.TVURoomAuthMode.CUSTOM);
        this.tvuSinglePlayerView = new TVUSinglePlayerView(applicationContext);
        InitConfig initConfig = new InitConfig();
        initConfig.singlePlayerListener = singlePlayerListener;
        tvuSinglePlayerView.init(initConfig);
        roomServer.setPlayerView(tvuSinglePlayerView.getInnerPlayerView());

        initConnect();
    }

    private void initConnect() {
        eventChannelHelper = new FlutterEventChannelHelper(binding.getBinaryMessenger(), BdliveFlutterViewerPlugin.LIVE_PLAYER_EVENT_CHANNEL + managerId);
        methodChannel = new MethodChannel(binding.getBinaryMessenger(), BdliveFlutterViewerPlugin.LIVE_PLAYER_CHANNEL + managerId);
        methodChannel.setMethodCallHandler(new MethodChannel.MethodCallHandler() {
            @Override
            public void onMethodCall(@NonNull MethodCall call, @NonNull MethodChannel.Result result) {
                Integer playerLayoutMode = call.argument("playerLayoutMode");
                Integer resolution = call.argument("resolution");
                Integer seekTimeInMills = call.argument("timeInMills");
                Boolean isMute = call.argument("isMute");
                Boolean isLoop = call.argument("isLoop");
                Double playSpeed = call.argument("playSpeed");
                Boolean inBackground = call.argument("inBackground");
                Boolean visible = call.argument("visible");


                switch (call.method) {
                    case "start":
                        roomServer.start();
                        invokeResult(result);
                        break;
                    case "setDelegate":
                        hasSetDelegate = true;
                        invokeResult(result);
                        break;
                    case "play":
                        tvuSinglePlayerView.play();
                        invokeResult(result);
                        break;
                    case "pause":
                        tvuSinglePlayerView.pause();
                        invokeResult(result);
                        break;
                    case "refreshLive":
                        tvuSinglePlayerView.refreshLive();
                        invokeResult(result);
                        break;
                    case "isPlaying":
                        result.success(tvuSinglePlayerView.isPlaying());
                        break;
                    case "isStalling":
                        result.success(tvuSinglePlayerView.isStalling());
                        break;
                    case "getPlayableStatus":
                        result.success(tvuSinglePlayerView.getPlayableStatus());
                        break;
                    case "setPlayerLayoutMode":
                        if (playerLayoutMode != null) {
                            tvuSinglePlayerView.setPlayerLayoutMode(playerLayoutMode);
                        }
                        invokeResult(result);
                        break;
                    case "getPlayerLayoutMode":
                        result.success(tvuSinglePlayerView.getPlayerLayoutMode());
                        break;
                    case "getResolutions":
                        List<Integer> resolutions = BDLVideoResolution.castStrList2ResolutionIntList(tvuSinglePlayerView.getResolutions());
                        result.success(resolutions);
                        break;
                    case "getCurResolution":
                        BDLVideoResolution videoResolution = BDLVideoResolution.castStr2BDLVideoResolution(tvuSinglePlayerView.getCurResolution());
                        if (videoResolution == null) {
                            videoResolution = BDLVideoResolution.BDLVideoResolutionUnknown;
                        }
                        result.success(videoResolution.value);
                        break;
                    case "setCurResolution":
                        BDLVideoResolution bdlVideoResolution = BDLVideoResolution.findBDLVideoResolution(resolution);
                        String string = resolutionStringHashMap.get(bdlVideoResolution);
                        if (!StringUtils.isEmpty(string)) {
                            tvuSinglePlayerView.setCurResolution(string);
                        }
                        invokeResult(result);
                        break;
                    case "getCurVodPlayTime":
                        result.success(tvuSinglePlayerView.getCurVodPlayTime());
                        break;
                    case "getCurVodDuration":
                        result.success(tvuSinglePlayerView.getCurVodDuration());
                        break;
                    case "seekVodTime":
                        if (seekTimeInMills != null) {
                            tvuSinglePlayerView.seekVodTime(seekTimeInMills, new SeekCompletionListener() {
                                @Override
                                public void onCompletion(boolean b) {
                                    result.success(b);
                                }
                            });
                        } else {
                            result.success(false);
                        }
                        break;
                    case "setMute":
                        if (isMute != null) {
                            tvuSinglePlayerView.setMute(isMute);
                        }
                        invokeResult(result);
                        break;
                    case "setVodLoop":
                        if (isLoop != null) {
                            tvuSinglePlayerView.setVodLoop(isLoop);
                        }
                        invokeResult(result);
                        break;
                    case "setVodPlaySpeed":
                        if (playSpeed != null) {
                            tvuSinglePlayerView.setVodPlaySpeed(playSpeed.floatValue());
                        }
                        invokeResult(result);
                        break;
                    case "getCurVodPlaySpeed":
                        result.success(tvuSinglePlayerView.getCurVodPlaySpeed());
                        break;
                    case "setInBackground":
                        if (inBackground != null) {
                            tvuSinglePlayerView.setInBackground(inBackground);
                        }
                        invokeResult(result);
                        break;
                    case "getLiveRoomStatus":
                        result.success(tvuSinglePlayerView.getLiveRoomStatus());
                        break;
                    case "setCoverImageContainerVisibility":
                        if (visible != null) {
                            tvuSinglePlayerView.setCoverImageContainerVisibility(visible ? View.VISIBLE : View.GONE);
                        }
                        invokeResult(result);
                        break;
                    case "destroy":
                        manager.destroyLivePlayer(managerId);
                        invokeResult(result);
                        break;
                    default:
                        result.notImplemented();
                        break;
                }
            }
        });
    }

    private void invokeResult(MethodChannel.Result result) {
        HashMap<String, Boolean> map = new HashMap<>();
        map.put("invokeResult", true);
        result.success(map);
    }

    public int getBindPlayerViewId() {
        return bindPlayerViewId;
    }

    public void setBindPlayerViewId(int bindPlayerViewId) {
        this.bindPlayerViewId = bindPlayerViewId;
    }

    private void sendEventToFlutter(String eventName, Map<String, Object> map) {
        if (!hasSetDelegate) {
            return;
        }
        map.put("methodName", eventName);
        eventChannelHelper.sendToFlutter(map);
    }

    public Integer getManagerId() {
        return managerId;
    }


    private int castBDLForceOfflineReason2Int(BDLForceOfflineReason reason) {
        int reasonInt = 0;
        switch (reason) {
            case BDLiveForceOfflineReasonMultiLogin:
                reasonInt = 0;
                break;
            case BDLiveForceOfflineReasonUnCheckIn:
                reasonInt = 1;
                break;
        }
        return reasonInt;
    }

    public TVUSinglePlayerView getTvuSinglePlayerView() {
        return tvuSinglePlayerView;
    }

    public void destroy() {
        methodChannel.setMethodCallHandler(null);
        eventChannelHelper.destroy();
        roomServer.closeRoom();
    }
}
