//
//  BdliveFlutterViewerPlugin.java
//
//  bdlive-flutter-viewer SDK License
//
//  Copyright 2025 Byteplus Pte. Ltd. All Rights Reserved.
//
//  The bdlive-flutter-viewer SDK was developed by Byteplus Pte. Ltd. (hereinafter “Byteplus”).
//  Any copyright or patent right is owned by and proprietary material of the Byteplus.
//
//  bdlive-flutter-viewer SDK is available under the Byteplus product and licensed under the commercial license.
//  Customers can contact via https://www.byteplus.com/en/contact for commercial licensing options.
//  Here is also a link to Service Level Agreement: https://docs.byteplus.com/en/byteplus-livesaas/docs/byteplus-live-service-level-agreement
//
//  Without Byteplus's prior written permission, any use of bdlive-flutter-viewer SDK, in particular any use for commercial purposes, is prohibited.
//  This includes, without limitation, incorporation in a commercial product, use in a commercial service, or production of other artefacts for commercial purposes.
//
//  Without Byteplus's prior written permission, the bdlive-flutter-viewer SDK may not be reproduced, modified and/or made available in any form to any third party.
package com.example.bdlive_flutter_viewer;

import android.app.Activity;
import android.content.Context;
import android.content.Intent;
import android.net.Uri;

import androidx.annotation.NonNull;

import com.bytedance.live.common.env.BDLiveConfig;
import com.bytedance.live.common.env.BDLiveEnv;
import com.bytedance.live.common.log.CommonLogger;
import com.bytedance.live.common.log.CustomLogger;
import com.bytedance.live.common.utils.IntentUtils;
import com.bytedance.live.common.utils.StringUtils;
import com.bytedance.live.sdk.BuildConfig;
import com.bytedance.live.sdk.player.CustomSettings;
import com.bytedance.live.sdk.player.TVULiveRoom;
import com.bytedance.live.sdk.player.TVULiveRoomServer;
import com.bytedance.live.sdk.player.activity.roomview.FlutterContainerHelper;
import com.bytedance.live.sdk.player.activity.roomview.IntentKey;
import com.bytedance.live.sdk.player.activity.roomview.JoinResultListener;
import com.bytedance.live.sdk.player.activity.roomview.JoinRoomConfig;
import com.bytedance.live.sdk.player.core.EnterRoomCallback;
import com.bytedance.live.sdk.player.core.EnterRoomManager;
import com.bytedance.live.sdk.player.core.JoinRoomData;
import com.bytedance.live.sdk.player.dialog.CommonWebViewDialog;
import com.bytedance.live.sdk.player.listener.FloatViewActionListener;
import com.bytedance.live.sdk.player.listener.RedirectPageListener;
import com.bytedance.live.sdk.player.logic.FloatManager;
import com.bytedance.live.sdk.util.ClickUtil;
import com.bytedance.live.sdk.util.FloatUtil;
import com.example.bdlive_flutter_viewer.livev.BDLiveFlutterPullViewFactory;
import com.example.bdlive_flutter_viewer.livev.BDLivePullView;
import com.example.bdlive_flutter_viewer.livev.BdlActivityContainer;
import com.example.bdlive_flutter_viewer.livev.FlutterContainerHelperContainer;
import com.example.bdlive_flutter_viewer.player.FlutterLivePlayerManager;
import com.example.bdlive_flutter_viewer.vo.BdlActivity;
import com.example.bdlive_flutter_viewer.vo.BdlViewerOpenUrlType;
import com.pandora.ttlicense2.LicenseManager;

import java.util.HashMap;
import java.util.Map;
import java.util.Set;
import java.util.UUID;

import io.flutter.Log;
import io.flutter.embedding.engine.plugins.FlutterPlugin;
import io.flutter.embedding.engine.plugins.activity.ActivityAware;
import io.flutter.embedding.engine.plugins.activity.ActivityPluginBinding;
import io.flutter.plugin.common.MethodCall;
import io.flutter.plugin.common.MethodChannel;
import io.flutter.plugin.common.MethodChannel.MethodCallHandler;
import io.flutter.plugin.common.MethodChannel.Result;

/**
 * BdliveVodPlayerPlugin
 */
public class BdliveFlutterViewerPlugin implements FlutterPlugin, MethodCallHandler, ActivityAware {
    private String TAG = this.getClass().getSimpleName();

    //Notify the initialized channel
    public static final String STATIC_CHANNEL = "com.bytedance.flutter_viewer.plugin";
    //Send Channel
    public static final String EVENT_CHANNEL_NAME = "com.bytedance.flutter_viewer.listener.event";

    //Live Player api channel
    public static final String LIVE_PLAYER_CHANNEL = "com.bytedance.live_player.";
    //Live Player event channel
    public static final String LIVE_PLAYER_EVENT_CHANNEL = "com.bytedance.live_player.delegate.event.";


    /// The MethodChannel that will the communication between Flutter and native Android
    ///
    /// This local reference serves to register the plugin with the Flutter Engine and unregister it
    /// when the Flutter Engine is detached from the Activity
    private MethodChannel channel;
    private FlutterPluginBinding flutterPluginBinding;
    private FlutterEventChannelHelper eventChannelHelper;
    private BdlActivity bdlActivity;
    private JoinRoomConfig joinRoomConfig;
    private boolean lastEnterFloatViewViaShowLivePullView=false;

    private Activity flutterActivity;

    Map<String, Integer> entranceValueMap = new HashMap<String, Integer>() {{
        put(RedirectPageListener.Entrance.HEADER_ADVERTISEMENT.value, BdlViewerOpenUrlType.pageHeaderAd.value);
        put(RedirectPageListener.Entrance.BANNER_ADVERTISEMENT.value, BdlViewerOpenUrlType.pageAd.value);
        put(RedirectPageListener.Entrance.FLOATING_ADVERTISEMENT.value, BdlViewerOpenUrlType.floatingAd.value);
        put(RedirectPageListener.Entrance.FLOATING_SHOPPING_CARD.value, BdlViewerOpenUrlType.cardAd.value);
    }};

    private final FlutterLiveRoomStatusListener roomStatusListener = new FlutterLiveRoomStatusListener() {
        @Override
        public void onLiveRoomActivityCreate(Activity activity) {

        }

        @Override
        public void onLiveRoomActivityResume(Activity activity) {
            sendEventToFlutter("onLiveRoomActivityResume", getBdlActivityMap());
        }

        @Override
        public void onLiveRoomActivityPause(Activity activity) {
            sendEventToFlutter("onLiveRoomActivityPause", getBdlActivityMap());
        }

        @Override
        public void onLiveRoomActivityDestroy(Activity activity) {

        }

        @Override
        public void onLiveRoomFullDestroyed(long l) {
            sendEventToFlutter("onLiveRoomFullDestroyed", getBdlActivityMap());
        }

        @Override
        public void onFloatViewWillAppear(long activityId) {
            sendEventToFlutter("onFloatViewWillAppear", getBdlActivityMap());
        }

        @Override
        public void onFloatViewWillClose(long activityId) {
            sendEventToFlutter("onFloatViewWillClose", getBdlActivityMap());
        }

        public void onShowLivePullView(long activityId){
            lastEnterFloatViewViaShowLivePullView=true;
            sendEventToFlutter("showLivePullView", getBdlActivityMap());
        }

        public void onHideLivePullView(long activityId){
            sendEventToFlutter("hideLivePullView", getBdlActivityMap());
        }

        private Map<String, Object> getBdlActivityMap() {
            Map<String, Object> map = new HashMap<>();
            if (bdlActivity != null) {
                map.put("activity", bdlActivity.generateFlutterMap());
            }
            return map;
        }
    };

    private FlutterLivePlayerManager flutterLivePlayerManager;

    @Override
    public void onAttachedToEngine(@NonNull FlutterPluginBinding flutterPluginBinding) {
        this.flutterPluginBinding = flutterPluginBinding;
        channel = new MethodChannel(flutterPluginBinding.getBinaryMessenger(), STATIC_CHANNEL);
        channel.setMethodCallHandler(this);
        eventChannelHelper = new FlutterEventChannelHelper(flutterPluginBinding.getBinaryMessenger(),
                EVENT_CHANNEL_NAME);

        flutterLivePlayerManager = new FlutterLivePlayerManager(flutterPluginBinding);

        CustomSettings.Holder.mSettings.setRedirectPageListener((activity, redirectInfo) -> {
            String entrance = redirectInfo.getEntrance();
            Set<String> strings = entranceValueMap.keySet();
            if (strings.contains(entrance)) {
                if (StringUtils.isEmpty(redirectInfo.getUrlRedirectInfo().getUrl())) {
                    //If the jump link is empty, it will not be processed.
                    return true;
                }
                Map<String, Object> map = new HashMap<>();
                map.put("url", redirectInfo.getUrlRedirectInfo().getUrl());
                map.put("type", entranceValueMap.get(entrance));
                map.put("enableFloating", redirectInfo.getUrlRedirectInfo().isEnableFloating());
                sendEventToFlutter("onOpenUrlCallback", map);
                return true;
            }
            return false;
        });
        BDLiveFlutterPullViewFactory.register(flutterPluginBinding);

    }

    @Override
    public void onMethodCall(@NonNull MethodCall call, @NonNull Result result) {
        Long activityId = call.argument("activityId");
        String token = call.argument("token");
        Integer authMode = call.argument("authMode");
        Boolean isPortrait = call.argument("isPortrait");

    
        switch (call.method) {
            case "startWithConfiguration":
                String appID = call.argument("appID");
                String appName = call.argument("appName");
                String channel = call.argument("channel");
                String liveLicenseFilePath = call.argument("liveLicenseFilePath");
                String vodLicenseFilePath = call.argument("vodLicenseFilePath");
                String appVersion = call.argument("appVersion");
                boolean isOversea = BuildConfig.IS_OVERSEA;
                if (StringUtils.isEmpty(channel)) {
                    channel = "BdliveFlutterViewerPlugin";
                }
                if (StringUtils.isEmpty(appVersion)) {
                    appVersion = "0.0.1";
                }
                BDLiveConfig bdLiveConfig = new BDLiveConfig.Builder()
                        .setApplicationContext(flutterPluginBinding.getApplicationContext())
                        .setAppId(appID)
                        .setAppName(appName)
                        .setAppChannel(channel)
                        .setAppVersion(appVersion)
                        .setVodLicenseUri(handleLicensePath(vodLicenseFilePath))
                        .setLiveLicenseUri(handleLicensePath(liveLicenseFilePath))
                        .setAppRegion(isOversea ? "singapore" : "china")
                        .setLicenseCallback(new LicenseManager.Callback() {
                            @Override
                            public void onLicenseLoadSuccess(@NonNull String licenseUri, @NonNull String licenseId) {
                                Log.d(TAG, "onLicenseLoadSuccess: " + licenseUri + "-" + licenseId);
                            }

                            @Override
                            public void onLicenseLoadError(@NonNull String licenseUri, @NonNull Exception e, boolean retryAble) {
                                Log.d(TAG, "onLicenseLoadError: " + licenseUri + "-" + e.getMessage());
                            }

                            @Override
                            public void onLicenseLoadRetry(@NonNull String licenseUri) {

                            }

                            @Override
                            public void onLicenseUpdateSuccess(@NonNull String licenseUri, @NonNull String licenseId) {

                            }

                            @Override
                            public void onLicenseUpdateError(@NonNull String licenseUri, @NonNull Exception e, boolean retryAble) {

                            }

                            @Override
                            public void onLicenseUpdateRetry(@NonNull String licenseUri) {

                            }
                        })
                        .build();

                BDLiveEnv.init(bdLiveConfig);
                CommonLogger.setChannelEnv(CommonLogger.channel_flutter_sdk);
                result.success("Init licenseManager");
                break;
            case "enterLiveRoom":
                bdlActivity = new BdlActivity(activityId, token, authMode, isPortrait);
                if (!bdlActivity.isValid()) {
                    result.error("-1", "args is not valid: " + bdlActivity, null);
                    return;
                }
                joinRoomConfig = new JoinRoomConfig(bdlActivity.getActivityId(), bdlActivity.getToken(),
                        bdlActivity.getPortrait(),
                        bdlActivity.getAuthMode() == 1 ? TVULiveRoom.TVURoomAuthMode.PUBLIC : TVULiveRoom.TVURoomAuthMode.CUSTOM,
                        new JoinResultListener() {
                            @Override
                            public void onJoinSuccess() {
                                result.success(true);
                            }

                            @Override
                            public void onJoinFailed(int i, String s) {
                                result.success(false);
                            }
                        });
                joinLiveRoom(flutterPluginBinding.getApplicationContext(), joinRoomConfig);
                lastEnterFloatViewViaShowLivePullView=false;
                break;
            case "leaveLiveRoom":
                TVULiveRoom.leaveLiveRoom(flutterPluginBinding.getApplicationContext());
                result.success(true);
                break;
            case "setLiveRoomStatusListener":
                CustomSettings.Holder.mSettings.setLiveRoomStatusListener(roomStatusListener);
                result.success(true);
                break;
            case "openUrl":
                String url = call.argument("url");
                Boolean floating = call.argument("enableFloating");
                boolean enableFloating = floating != null && floating;
                if (enableFloating) {
                    TVULiveRoomServer server = TVULiveRoomServer.Holder.getServer();
                    if (server != null) {
                        Context context = server.getContext();
                        if (context instanceof Activity) {
                            CommonWebViewDialog mCommonWebViewDialog = new CommonWebViewDialog(context,
                                    url,
                                    com.bytedance.live.sdk.R.style.TvuLiveBottomDialog);
                            mCommonWebViewDialog.setUrl(url);
                            mCommonWebViewDialog.show();
                        }
                    }
                } else {
                    IntentUtils.attemptViewUri(flutterPluginBinding.getApplicationContext(),
                            Uri.parse(url));
                }
                result.success(true);
                break;
            case "createPlayer":
                Integer managerId = call.argument("managerId");
                BdlActivity activity = new BdlActivity(activityId, token, authMode, isPortrait);
                if (activity.isValid() && managerId != null) {
                    flutterLivePlayerManager.createLivePlayer(managerId, activity);
                }
                result.success(true);
                break;
            case "hideLiveRoom":
                if (flutterActivity != null) {
                    Intent intent = new Intent(flutterPluginBinding.getApplicationContext(), flutterActivity.getClass());
                    // 如果 Activity 已经在运行，则不会创建新的实例，而是将其带到前台。
                    intent.setFlags(Intent.FLAG_ACTIVITY_REORDER_TO_FRONT);
                    flutterActivity.startActivity(intent);
                }
                result.success(true);
                break;
            case "setLiveRoomConfig":
                Map<String, Object> config = call.argument("playerConfig");
                if (config != null) {
                    CustomSettings.Holder.mSettings.setLiveAutoPlay((Boolean) config.get("liveAutoPlay"));
                    CustomSettings.Holder.mSettings.setForceShowAutoPlay((Boolean) config.get("previewAutoPlay"));
                    CustomSettings.Holder.mSettings.setPlayBackAutoPlay((Boolean) config.get("replayAutoPlay"));
                    CustomSettings.Holder.mSettings.setOpenLiveFloatWindow((Boolean) config.get("enablePIP"));
                    CustomSettings.Holder.mSettings.setOpenVodFloatWindow((Boolean) config.get("enablePIP"));
                    CustomSettings.Holder.mSettings.setOpenFloatWindowWhenExit((Boolean) config.get("enablePIPWhenExit"));
                }
                result.success(true);
                break;
            case "enterLiveRoomLogically":
                bdlActivity = new BdlActivity(activityId, token, authMode, isPortrait);
                if (!bdlActivity.isValid()) {
                    result.success(false);
                    return;
                }
                BdlActivityContainer.setBdlActivity(bdlActivity);
                EnterRoomManager.getInstance().enterRoom(UUID.randomUUID().toString(),
                        bdlActivity.getActivityId(), bdlActivity.getToken(),
                        bdlActivity.getAuthMode() ==2 ? TVULiveRoom.TVURoomAuthMode.CUSTOM:
                        TVULiveRoom.TVURoomAuthMode.PUBLIC, new EnterRoomCallback() {
                    @Override
                    public void onSuccess(String flowId, JoinRoomData joinRoomData) {
                        result.success(true);
                        roomStatusListener.onShowLivePullView(bdlActivity.getActivityId());
                    }

                    @Override
                    public void onFailed(String flowId, int errorCode, String errorMessage) {
                        result.success(false);
                    }
                });

                break;
            case "dispatchWidgetSate":
                Integer stateValue=call.argument("state");
                FlutterContainerHelper flutterContainerHelper=FlutterContainerHelperContainer.getFlutterContainerHelper();
                if (flutterContainerHelper!=null && stateValue!=null){
                    WidgetState widgetState = WidgetState.fromValue(stateValue);
                    switch (widgetState){
                        case RESUME:
                            flutterContainerHelper.onFlutterPageResume();
                            break;
                        case PAUSE:
                            flutterContainerHelper.onFlutterPagePause();
                            break;
                        case DESTROY:
                            flutterContainerHelper.onFlutterPageDestroy();
                            Log.w("LIFECYCLE", "dispatchWidgetSate: onFlutterPageDestroy");
                            break;
                        default:
                            break;
                    }
                }
                result.success(null);
                break;
            case "leaveLiveRoomLogically":
                EnterRoomManager.getInstance().leaveRoom(BDLivePullView.enterID);
                TVULiveRoom.leaveLiveRoom(flutterPluginBinding.getApplicationContext());
                result.success(true);
                break;
            case "onWillClose":
                FlutterContainerHelper flutterContainer = FlutterContainerHelperContainer.getFlutterContainerHelper();
                if(flutterContainer==null){
                    result.success(false);
                    return;
                }
                result.success(flutterContainer.onFlutterActivityExitKeyDown());
                break;
            default:
                result.notImplemented();
                break;
        }
    }

    public Activity getFlutterActivity() {
        return flutterActivity;
    }

    private void sendEventToFlutter(String eventName, Map<String, Object> map) {
        map.put("methodName", eventName);
        eventChannelHelper.sendToFlutter(map);
    }

    private String handleLicensePath(String path) {
        if (StringUtils.isEmpty(path)) {
            return path;
        }
        if (!path.startsWith("http://") &&
                !path.startsWith("https://")) {
            path = "assets:///flutter_assets/" + path;
        }
        return path;
    }

    /**
     * @param context        上下文
     * @param joinRoomConfig 进房配置信息
     */
    public static void joinLiveRoom(final Context context, JoinRoomConfig joinRoomConfig) {
        if (ClickUtil.isFastDoubleClick("TVULiveRoom-joinLiveRoom", 1000)) {
            //1s内仅支持调用一次
            return;
        }
        CustomLogger.apiCallLog(TVULiveRoom.class, "joinLiveRoom",
                joinRoomConfig.activityId, joinRoomConfig.isPortraitLiveRoom, joinRoomConfig.authMode, joinRoomConfig.joinResultListener,
                StringUtils.stringObfuscation(joinRoomConfig.token));
        Intent intent;
        String requestId = UUID.randomUUID().toString();
        intent = new Intent(context, FlutterLiveRoomActivity.class);
        intent.setFlags(Intent.FLAG_ACTIVITY_NEW_TASK);
        intent.putExtra(IntentKey.INTENT_ACTIVITY_ID, joinRoomConfig.activityId);
        intent.putExtra(IntentKey.INTENT_TOKEN, joinRoomConfig.token);
        intent.putExtra(IntentKey.INTENT_ROOM_AUTH_MODE, joinRoomConfig.authMode);
        intent.putExtra(IntentKey.INTENT_IS_PORTRAIT_LIVEROOM, joinRoomConfig.isPortraitLiveRoom);
        if (joinRoomConfig.joinResultListener != null) {
            intent.putExtra(IntentKey.INTENT_REQUEST_ID, requestId);
            FlutterLiveRoomActivity.getJoinResultListenerMap().put(requestId, joinRoomConfig.joinResultListener);
        }
        context.startActivity(intent);
    }

    @Override
    public void onDetachedFromEngine(@NonNull FlutterPluginBinding binding) {
        channel.setMethodCallHandler(null);
        eventChannelHelper.destroy();
        CustomSettings.Holder.mSettings.setLiveRoomStatusListener(null);
        CustomSettings.Holder.mSettings.setRedirectPageListener(null);
        flutterLivePlayerManager.destroy();
        if(FloatManager.getCurInstance()!=null){
            FloatManager.getCurInstance().destroyFloatWindow();
        }

    }

    @Override
    public void onAttachedToActivity(@NonNull ActivityPluginBinding binding) {
        Log.d(TAG, "onAttachedToActivity");
        flutterActivity = binding.getActivity();
        Activity local=binding.getActivity();
        FlutterContainerHelperContainer.setFlutterContainerHelper(new FlutterContainerHelper(flutterActivity){
            @Override
            public boolean onHandleContainerFinish() {
                if(BdlActivityContainer.getBdlActivity()!=null){
                    roomStatusListener.onHideLivePullView(BdlActivityContainer.getBdlActivity().getActivityId());
                }
                return super.onHandleContainerFinish();
            }

            @Override
            public void onFlutterPageDestroy() {
                super.onFlutterPageDestroy();
            }
        });

        CustomSettings.Holder.mSettings.setFloatViewActionListener(new FloatViewActionListener() {
            @Override
            public boolean onClickFloatView(long l) {
                if(!lastEnterFloatViewViaShowLivePullView){
                    return false;
                }
                if(local!=null){
                    Intent bindActivityIntent= local.getIntent();
                    bindActivityIntent.setFlags(Intent.FLAG_ACTIVITY_NEW_TASK);
                    local.getApplication().startActivity(bindActivityIntent);
                    FloatUtil.setTopApp(local.getApplicationContext());
                    roomStatusListener.onShowLivePullView(BdlActivityContainer.getBdlActivity().getActivityId());
                    return true;
                }
                return false;
            }

            @Override
            public boolean onClickFloatViewExitBtn(long l) {
                return false;
            }
        });
    }

    @Override
    public void onDetachedFromActivityForConfigChanges() {

    }

    @Override
    public void onReattachedToActivityForConfigChanges(@NonNull ActivityPluginBinding binding) {

    }

    @Override
    public void onDetachedFromActivity() {
        Log.d(TAG, "onDetachedFromActivity");
        if(FlutterContainerHelperContainer.getFlutterContainerHelper()!=null&&flutterActivity==FlutterContainerHelperContainer.getFlutterContainerHelper().getActivity()){
            FlutterContainerHelperContainer.setFlutterContainerHelper(null);
        }
        flutterActivity = null;
    }
}
